# coding: utf-8


import logging
from collections import defaultdict

from django.conf import settings
from django.db import connection
from django.db.models import NOT_PROVIDED
from django.utils.translation import ugettext_noop as _
from django_pgaas.transaction import atomic_retry

from idm.core.constants.system import SYSTEM_GROUP_POLICY, SYSTEM_AUDIT_METHOD
from idm.core.constants.rolenode import ROLENODE_STATE
from idm.users.models import User, Group
from idm.utils.cleansing import cleanup_fields_data
from idm.core.plugins.generic_new import Plugin as NewPlugin
from idm.inconsistencies.models import Inconsistency

log = logging.getLogger(__name__)

INC_STATE = Inconsistency.STATE
INC_TYPE = Inconsistency.TYPE

__all__ = ('check_roles', 'BatchInserter', 'get_roles', 'stream_roles')


def other_as_dict(value):
    result = None
    if isinstance(value, dict):
        result = {
            'data': value
        }
    elif isinstance(value, (list, tuple)):
        if len(value) == 1:
            result = {
                'data': value[0],
            }
        elif len(value) == 2:
            role, fields_data = value
            result = {
                'data': role,
                'fields': cleanup_fields_data(fields_data)
            }
    return result


class BatchInserter:
    def __init__(self, batch_size):
        self.batch_size = batch_size
        self.batch = []
        self.inconsistency_model = Inconsistency

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            if len(self.batch):
                self.flush()

    def insert(self, inconsistency):
        self.batch.append(inconsistency)
        if len(self.batch) >= self.batch_size:
            self.flush()

    def flush(self):
        self.inconsistency_model.objects.bulk_create(self.batch, ignore_conflicts=True)
        self.batch = []


def get_roles(system):
    method = getattr(system.plugin, system.audit_method)
    try:
        plugin_roles = method()
    except Exception:
        log.exception('Exception while fetching roles with %s from "%s"', system.audit_method, system.slug)
        plugin_roles = False
    return system.audit_method, plugin_roles


def stream_roles(all_roles):
    for fieldname, owner_roles in (
        ('uid', all_roles.get('users', [])),
        ('login', all_roles.get('users', [])),
        ('group', all_roles.get('groups', []))
    ):
        for owner_role in owner_roles:
            if fieldname not in owner_role:
                continue
            ident = owner_role[fieldname]
            subject_type = owner_role.get('subject_type') or None
            roles = owner_role['roles']
            for role in roles:
                role_dict = other_as_dict(role)
                if role_dict is None:
                    continue
                role_dict[fieldname] = ident
                if subject_type:
                    role_dict['subject_type'] = subject_type
                yield role_dict


SQL_LINK_NODES_BY_DATA = f'''
UPDATE upravlyator_inconsistency AS ic
SET node_id = rn.id
FROM upravlyator_rolenode rn
WHERE (
  ic.remote_data = rn.data AND
  ic.ident_type = 'data' AND
  ic.system_id = %s AND
  rn.system_id = %s AND
  ic.sync_key_id = %s AND
  ic.state IN ('{INC_STATE.CREATED}', '{INC_STATE.ACTIVE}') AND
  rn.state = %s
);'''


SQL_LINK_NODES_BY_PATH = f'''
UPDATE upravlyator_inconsistency AS ic
SET node_id = rn.id
FROM upravlyator_rolenode rn 
WHERE (
  ic.remote_path = rn.slug_path AND
  ic.ident_type = 'path' AND
  ic.system_id = %s AND
  rn.system_id = %s AND
  ic.sync_key_id = %s AND
  ic.state IN ('{INC_STATE.CREATED}', '{INC_STATE.ACTIVE}') AND
  rn.state = %s
);
'''

SQL_AWARE_MATCH_GROUP_ROLES = f'''
INSERT INTO upravlyator_matching_role (inconsistency_id, role_id)
SELECT ic.id, r.id FROM upravlyator_inconsistency ic
INNER JOIN upravlyator_role r ON (
  ic.node_id=r.node_id AND
  ic.group_id=r.group_id AND r.group_id IS NOT NULL AND
  (
    (r.system_specific=ic.remote_fields) OR (r.system_specific IS NULL AND ic.remote_fields IS NULL)
  )
  {{plugin_condition}}
)
WHERE
  (r.is_active OR r.updated > %s) AND
  r.system_id=%s AND
  ic.system_id=%s AND
  ic.sync_key_id=%s AND
  ic.state='{INC_STATE.CREATED}' AND
  ic.node_id IS NOT NULL;
'''

SQL_AWARE_MATCH_GROUP_ROLES_NEW_PLUGIN_CONDITION = '''  
  AND ic.with_inheritance=r.with_inheritance AND
  ic.with_robots=r.with_robots AND
  ic.with_external=r.with_external
'''

SQL_MATCH_USER_ROLES = f'''
INSERT INTO upravlyator_matching_role (inconsistency_id, role_id)
SELECT ic.id, r.id FROM upravlyator_inconsistency ic
INNER JOIN upravlyator_role r ON (
  ic.node_id=r.node_id AND ic.user_id=r.user_id AND r.user_id IS NOT NULL AND
  (
    (r.system_specific=ic.remote_fields) OR (r.system_specific IS NULL AND ic.remote_fields IS NULL)
  )
)
WHERE (r.is_active OR r.updated > %s) AND
      r.system_id=%s AND
      ic.system_id=%s AND
      ic.sync_key_id=%s AND
      ic.state='{INC_STATE.CREATED}' AND
      ic.node_id IS NOT NULL;
'''


SQL_UPDATE_MISMATCHING_INCONSISTENCIES = f'''
UPDATE upravlyator_inconsistency ic
SET state = '{INC_STATE.ACTIVE}'
WHERE id IN (
  SELECT ic.id FROM upravlyator_inconsistency ic
  LEFT OUTER JOIN upravlyator_matching_role mr ON (ic.id=mr.inconsistency_id)
  WHERE mr.id IS NULL AND ic.sync_key_id=%s
);
'''

SQL_UPDATE_MATCHING_INCONSISTENCIES = f'''
UPDATE upravlyator_inconsistency ic
SET type = '{INC_STATE.MATCHING}'
WHERE id IN (
  SELECT ic.id FROM upravlyator_inconsistency ic
  LEFT OUTER JOIN upravlyator_matching_role mr ON (ic.id=mr.inconsistency_id)
  WHERE mr.id IS NOT NULL AND ic.sync_key_id=%s
);
'''


SQL_AWARE_CREATE_OUR_SIDE_INCONSISTENCIES = f'''
INSERT INTO upravlyator_inconsistency (
  system_id, sync_key_id, state, ident_type,
  user_id, group_id, node_id, is_forced,
  our_role_id, type, added, updated
)
SELECT
  %s, %s, '{INC_STATE.ACTIVE}', 'data',
  r.user_id, r.group_id, r.node_id, FALSE,
  r.id, '{INC_TYPE.OUR}', CURRENT_TIMESTAMP, CURRENT_TIMESTAMP
FROM upravlyator_role r
  INNER JOIN upravlyator_system s ON (r.system_id=s.id)
  LEFT OUTER JOIN (
    SELECT m.id, m.role_id from upravlyator_inconsistency i 
    INNER JOIN upravlyator_matching_role m 
    ON i.id=m.inconsistency_id
    WHERE i.sync_key_id=%s
  ) as mr ON (mr.role_id=r.id)
  WHERE r.system_id=%s AND r.is_active AND mr.id IS NULL AND r.updated < %s
ORDER BY r.id;
'''


SQL_UNAWARE_CREATE_OUR_SIDE_INCONSISTENCIES = f'''
INSERT INTO upravlyator_inconsistency (
  system_id, sync_key_id, state, ident_type,
  user_id, group_id, node_id, is_forced,
  our_role_id, type, added, updated,
  remote_hash
)
SELECT
  %s, %s, '{INC_STATE.ACTIVE}', 'data',
  r.user_id, NULL, r.node_id, FALSE,
  r.id, '{INC_TYPE.OUR}', CURRENT_TIMESTAMP, CURRENT_TIMESTAMP,
  md5(concat(r.user_id, r.node_id, r.fields_data))
FROM upravlyator_role r
  INNER JOIN upravlyator_system s ON (r.system_id=s.id)
  LEFT OUTER JOIN (
    SELECT m.id, m.role_id from upravlyator_inconsistency i 
    INNER JOIN upravlyator_matching_role m 
    ON i.id=m.inconsistency_id
    WHERE i.sync_key_id=%s
  ) as mr ON (mr.role_id=r.id)
WHERE r.system_id=%s AND r.is_active AND mr.id IS NULL AND r.group_id IS NULL 
AND r.updated < %s
ORDER BY r.parent_id, r.id NULLS LAST
ON CONFLICT DO NOTHING;
'''


SQL_DELETE_MATCHING_ROLES = '''
DELETE FROM upravlyator_matching_role
WHERE id IN (
  SELECT mr.id
  FROM upravlyator_matching_role mr
  JOIN upravlyator_inconsistency ic ON ic.id=mr.inconsistency_id
  WHERE ic.sync_key_id=%s
);
'''


SQL_DELETE_MATCHING_INCONSISTENCIES = f'''
DELETE FROM upravlyator_inconsistency
WHERE system_id = %s AND sync_key_id = %s AND type = '{INC_STATE.MATCHING}';
'''


class InconsistencyProcessor(object):
    def __init__(self, system, action):
        self.system = system
        self.action = action

    def process(self):
        # 1. Линкуем узлы по data или path
        self.link_by_identity()
        # 2. Неслинкованным неконсистентностям проставим тип "неизвестный узел"
        self.set_unknown_node_for_unlinked()
        # 3. Оставшимся неконсистентностям (слинкованным, но пока неизвестного типа) проставим тип "на стороне системы"
        self.set_their_for_linked()
        # 4. Для тех ролей, которые должны быть отражены в выгрузке системы, создаём записи в таблице matching_role
        self.export_matching_roles()
        # 5. Тем неконсистентностям, для которых не создалось ни одной matching role, проставим статус 'active'
        self.set_nonmatching_as_active()
        # 6. Оставшимся неконсистентностям (совпадающим) проставим статус 'matching'. Чуть позже мы удалим их.
        self.set_matching_as_matching()
        # 7. Для всех ролей, у которых не нашлось совпадающих matching role, создадим неконсистентности типа
        # "у нас есть, в системе нет"
        self.create_ours()
        # 8. Удалим все matching role этой синхронизации +
        # 9. Удалим неконсистентности-совпадения
        self.drop_matching()

    @atomic_retry
    def link_by_identity(self):
        # в get_all_roles система отдает data, в get_roles отдает path
        if self.system.audit_method == SYSTEM_AUDIT_METHOD.GET_ALL_ROLES:
            link_query = SQL_LINK_NODES_BY_DATA
        else:
            link_query = SQL_LINK_NODES_BY_PATH
        with connection.cursor() as cursor:
            for node_state in (ROLENODE_STATE.DEPRIVING, ROLENODE_STATE.ACTIVE):
                cursor.execute(link_query, [self.system.pk, self.system.pk, self.action.pk, node_state])

    @atomic_retry
    def set_unknown_node_for_unlinked(self):
        Inconsistency.objects.filter(
            system=self.system,
            sync_key=self.action,
            state=INC_STATE.CREATED,
            type=INC_TYPE.UNDECIDED_YET,
            node=None
        ).update(type=INC_TYPE.UNKNOWN_ROLE, state=INC_STATE.ACTIVE)

    @atomic_retry
    def set_their_for_linked(self):
        Inconsistency.objects.filter(
            system=self.system,
            sync_key=self.action,
            state=INC_STATE.CREATED,
            type=INC_TYPE.UNDECIDED_YET
        ).update(type=INC_TYPE.THEIR)

    @atomic_retry
    def export_matching_roles(self):
        with connection.cursor() as cursor:
            if self.system.group_policy in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS:
                sql = SQL_AWARE_MATCH_GROUP_ROLES.format(
                    plugin_condition=SQL_AWARE_MATCH_GROUP_ROLES_NEW_PLUGIN_CONDITION
                    if isinstance(self.system.plugin, NewPlugin) else ''
                )
                # выгрузим совпадающие групповые роли
                cursor.execute(sql, [
                    self.action.added,
                    self.system.pk,
                    self.system.pk,
                    self.action.pk
                ])

            # выгрузим все совпадающие пользовательские роли
            cursor.execute(SQL_MATCH_USER_ROLES, [
                self.action.added,
                self.system.pk,
                self.system.pk,
                self.action.pk
            ])

    @atomic_retry
    def set_nonmatching_as_active(self):
        with connection.cursor() as cursor:
            cursor.execute(SQL_UPDATE_MISMATCHING_INCONSISTENCIES, [self.action.pk])

    @atomic_retry
    def set_matching_as_matching(self):
        with connection.cursor() as cursor:
            cursor.execute(SQL_UPDATE_MATCHING_INCONSISTENCIES, [self.action.pk])

    @atomic_retry
    def create_ours(self):
        with connection.cursor() as cursor:
            if self.system.group_policy in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS:
                cursor.execute(SQL_AWARE_CREATE_OUR_SIDE_INCONSISTENCIES, [
                    self.system.pk,
                    self.action.pk,
                    self.action.pk,
                    self.system.pk,
                    self.action.added
                ])
            else:
                cursor.execute(SQL_UNAWARE_CREATE_OUR_SIDE_INCONSISTENCIES, [
                    self.system.pk,
                    self.action.pk,
                    self.action.pk,
                    self.system.pk,
                    self.action.added
                ])

    @atomic_retry
    def drop_matching(self):
        with connection.cursor() as cursor:
            cursor.execute(SQL_DELETE_MATCHING_ROLES, [self.action.pk])
        with connection.cursor() as cursor:
            cursor.execute(SQL_DELETE_MATCHING_INCONSISTENCIES, [self.system.pk, self.action.pk])


def check_roles(system, threshold, action):
    if threshold is NOT_PROVIDED:
        threshold = (
            system.inconsistencies_for_break
            if system.inconsistencies_for_break is not None
            else settings.IDM_SYSTEM_BREAKDOWN_INCONSYSTENCY_COUNT
        )

    log.info('Checking roles in {0}'.format(system.slug))
    method_name, plugin_roles = get_roles(system)
    if plugin_roles is NotImplemented:
        return True
    elif plugin_roles is False:
        return False
    if method_name == SYSTEM_AUDIT_METHOD.GET_ROLES:
        ident_type = 'path'
        all_roles = plugin_roles
    elif method_name == SYSTEM_AUDIT_METHOD.GET_ALL_ROLES:
        ident_type = 'data'
        all_roles = stream_roles(plugin_roles)
    else:
        raise ValueError(method_name)

    users = {user.username: user for user in User.objects.all()}
    users_per_uid = {user.uid: user for user in users.values() if user.uid}
    groups = {}

    for group in Group.objects.user_groups():
        if group.is_active() or group.external_id not in groups:
            groups[group.external_id] = group

    # узнаем, сколько до прогона было неконсистентностей логина/группы
    unknown_subjects_count = system.inconsistencies.filter(
        state=INC_STATE.ACTIVE, type__in=(INC_TYPE.UNKNOWN_GROUP, INC_TYPE.UNKNOWN_USER)
    ).count()
    # закроем все старые неконсистентности
    system.inconsistencies.filter(state=INC_STATE.ACTIVE).update(state=INC_STATE.OBSOLETE)

    with BatchInserter(settings.IDM_INCONSISTENCY_BATCH_SIZE) as inserter:
        for role in all_roles:
            username = group_id = user = uid = group = path = data = None
            type_ = INC_TYPE.UNDECIDED_YET
            state = INC_STATE.CREATED
            if 'uid' in role:
                uid = role['uid']
                if uid in users_per_uid:
                    user = users_per_uid[uid]
                else:
                    type_ = INC_TYPE.UNKNOWN_USER
                    state = INC_STATE.ACTIVE
                    username = f'uid:{uid}'
            elif 'login' in role:
                username = role['login'].lower().strip()
                if username in users:
                    user = users[username]
                else:
                    type_ = INC_TYPE.UNKNOWN_USER
                    state = INC_STATE.ACTIVE
            elif 'group' in role:
                if system.group_policy not in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS:
                    log.warning('%s system %s audit response contains group roles', system.group_policy, system.slug)
                    continue
                group_id = int(role['group'])
                if group_id in groups:
                    group = groups[group_id]
                else:
                    type_ = INC_TYPE.UNKNOWN_GROUP
                    state = INC_STATE.ACTIVE
            else:
                keys = ','.join(map(str, role.keys()))
                log.warning('System %s responded with unknown roles type: %s', system.slug, keys)
                continue
            if ident_type == 'path':
                path = role['path']
            else:
                data = role['data']

            fields = role.get('fields') or None
            subject_type = role.get('subject_type') or None
            inconsistency = Inconsistency(
                system=system,
                sync_key=action,
                state=state,
                ident_type=ident_type,
                remote_username=username,
                remote_uid=uid,
                remote_group=group_id,
                remote_path=path,
                remote_data=data,
                remote_fields=fields,
                remote_subject_type=subject_type,
                user=user,
                group=group,
                type=type_,
                with_inheritance=role.get('with_inheritance'),
                with_robots=role.get('with_robots'),
                with_external=role.get('with_external'),
            )
            inconsistency.remote_hash = inconsistency.calculate_hash()
            inserter.insert(inconsistency)
    try:
        processor = InconsistencyProcessor(system, action)
        processor.process()
    except Exception as e:
        log.exception('Error while processing inconsistencies')
        raise

    active_inconsistencies = Inconsistency.objects.filter(sync_key=action, system=system, state=INC_STATE.ACTIVE)
    inconsistency_count = active_inconsistencies.count_to_break_system(system, unknown_subjects_count)
    log.info('got %d inconsistencies in system %s', inconsistency_count, system.slug)

    if (system.can_be_broken and
        system.inconsistency_policy != 'trust' and
        threshold is not None and
        inconsistency_count > threshold
    ):
        active_inconsistencies = active_inconsistencies.select_related('user', 'group', 'node')
        unknown_usernames = active_inconsistencies.filter(type=INC_TYPE.UNKNOWN_USER)
        unknown_groups = active_inconsistencies.filter(type=INC_TYPE.UNKNOWN_GROUP)
        we_have_system_dont = active_inconsistencies.filter(type=INC_TYPE.OUR)
        system_has_we_dont = active_inconsistencies.filter(type=INC_TYPE.THEIR)
        unknown_roles = active_inconsistencies.filter(type=INC_TYPE.UNKNOWN_ROLE)

        log.warning('blocking system %s', system.slug)
        we_have = defaultdict(list)
        for inconsistency in we_have_system_dont:
            we_have[inconsistency.get_owner()].append(inconsistency.node)
        system_has = defaultdict(list)
        for inconsistency in system_has_we_dont:
            system_has[inconsistency.get_owner()].append(inconsistency.node)
        system_has_unknown_roles = defaultdict(list)
        for inconsistency in unknown_roles:
            system_has_unknown_roles[inconsistency.get_owner()].append(inconsistency)
        context = {
            'we_have_system_dont': list(we_have.items()),
            'system_has_we_dont': list(system_has.items()),
            'unknown_usernames': unknown_usernames.values_list('remote_username', flat=True),
            'unknown_groups': unknown_groups.values_list('remote_group', flat=True),
            'unknown_roles': list(system_has_unknown_roles.items()),
            'total_count': inconsistency_count,
        }
        system.set_broken(
            reason=_('Слишком много расхождений: %s') % inconsistency_count,
            context=context,
        )

    return True
