from collections import defaultdict, namedtuple
import logging
import time
from typing import *
import ujson as json

from django.conf import settings
from django.db.models import NOT_PROVIDED, Value, IntegerField, Q
from django.utils.translation import ugettext_noop as _
from ylog.context import log_context

from idm.core.constants.system import SYSTEM_GROUP_POLICY, SYSTEM_AUDIT_METHOD
from idm.core.constants.rolenode import ROLENODE_STATE
from idm.core.constants.inconsistency_audit import INCONSISTENCY_AUDIT_STEPS
from idm.inconsistencies.audit import BatchInserter, get_roles, stream_roles
from idm.inconsistencies.models import Inconsistency
from idm.core.models import RoleNode
from idm.users.models import User, Group
from idm.core.plugins.generic_new import Plugin as NewPlugin

log = logging.getLogger(__name__)

INC_STATE = Inconsistency.STATE
INC_TYPE = Inconsistency.TYPE

RemoteRole = namedtuple(
    'RemoteRole',
    ['user', 'group', 'subject_type', 'identity', 'fields', 'with_inheritance', 'with_robots', 'with_external'],
)


class InconsistencyProcessor(object):
    def __init__(
        self,
        system: Any,  # System
        action: Any,  # Action
        ident_type: str,
        remote_roles: List[RemoteRole],
        index_to_identity: Dict[int, str],
        user_id_to_username: Dict[int, str],
        group_id_to_external_id: Dict[int, int],
    ):
        self.system = system
        self.action = action
        self.ident_type = ident_type
        self.remote_roles = remote_roles  # роли в системе
        self.remote_roles_set = set()
        self.index_to_identity = index_to_identity
        self.identity_to_node_id = {}  # user.slug_path/data -> user.id
        self.user_id_to_username = user_id_to_username  # user.id -> user.username
        self.group_id_to_external_id = group_id_to_external_id  # group.id -> group.external_id

    def _create_inconsistency(
            self,
            description: RemoteRole,
            inc_type: str,
            override_identity: Optional[str] = None
    ) -> Any:  # Inconsistency
        identity = description.identity
        if override_identity is not None:
            identity = override_identity
        node_id = None
        path = None
        data = None
        if inc_type in (INC_TYPE.OUR, INC_TYPE.THEIR):
            node_id = identity
            if inc_type != INC_TYPE.OUR:
                effective_identity = self.node_id_to_identity[node_id]
                path = effective_identity if self.ident_type == 'path' else None
                data = json.loads(effective_identity) if self.ident_type == 'data' else None
        else:
            path = identity if self.ident_type == 'path' else None
            data = json.loads(identity) if self.ident_type == 'data' else None

        if inc_type == INC_TYPE.OUR:
            # Для консистентности со старым механизмом сверки:
            # remote_username и remote_group проставляются только у расхождений, приехавших извне
            remote_username = None
            remote_group = None
        else:
            remote_username = (description.user and self.user_id_to_username[description.user])
            remote_group = (description.group and self.group_id_to_external_id[description.group])

        inconsistency = Inconsistency(
            system=self.system,
            sync_key=self.action,
            state=INC_STATE.ACTIVE,
            ident_type=self.ident_type,
            remote_username=remote_username,
            remote_group=remote_group,
            remote_path=path,
            remote_data=data,
            remote_fields=((description.fields and json.loads(description.fields)) if inc_type != INC_TYPE.OUR else None),
            remote_subject_type=description.subject_type,
            user_id=description.user,
            group_id=description.group,
            node_id=node_id,
            our_role_id=(self.active_roles_lower[description] if inc_type == INC_TYPE.OUR else None),
            type=inc_type,
            with_inheritance=description.with_inheritance,
            with_robots=description.with_robots,
            with_external=description.with_external,
        )
        if inc_type != INC_TYPE.OUR:
            inconsistency.remote_hash = inconsistency.calculate_hash()
        return inconsistency

    def process(self) -> None:
        with log_context(system=self.system.slug, source='audit_memory', unique_id=self.action.pk):
            for step in INCONSISTENCY_AUDIT_STEPS.INCONSISTENCY_PROCESSOR_STEPS:
                start = time.monotonic()
                with log_context(step=step):
                    getattr(self, step)()
                    with log_context(total_time=time.monotonic() - start, audit_step_finished=True):
                        log.info('Finished %s for system %s', step, self.system.slug)

    def link_by_identity(self) -> None:
        qset = RoleNode.objects.filter(system=self.system)
        if self.system.audit_method == SYSTEM_AUDIT_METHOD.GET_ALL_ROLES:
            nodes = qset.values_list('id', 'state', 'data')
        else:
            nodes = qset.values_list('id', 'state', 'slug_path')

        self.node_id_to_identity = {}
        # Порядок важен: если есть одинаковые depriving и active узлы, то отзываемый узел будет перезаписан активным,
        # так как активные узлы в большем приоритете
        for current_state in (ROLENODE_STATE.DEPRIVED, ROLENODE_STATE.DEPRIVING, ROLENODE_STATE.ACTIVE):
            for node_id, state, identity in nodes:
                if state != current_state:
                    continue
                identity = json.dumps(identity, sort_keys=True) if self.system.audit_method == SYSTEM_AUDIT_METHOD.GET_ALL_ROLES else identity
                self.node_id_to_identity[node_id] = identity
                # deprived узлы не нужно разрешать из identity
                if current_state != ROLENODE_STATE.DEPRIVED:
                    self.identity_to_node_id[identity] = node_id

    def update_unknown_subject_nodes(self) -> None:
        inconsistencies = Inconsistency.objects.filter(
            state=INC_STATE.ACTIVE,
            sync_key=self.action,
            type__in=(INC_TYPE.UNKNOWN_GROUP, INC_TYPE.UNKNOWN_USER)
        )
        for inc in inconsistencies:
            if inc.remote_data is not None:
                identity = json.dumps(inc.remote_data, sort_keys=True)
            else:
                identity = inc.remote_path
            node_id = self.identity_to_node_id.get(identity)
            if node_id is not None:
                inc.node_id = node_id
                inc.save(update_fields=['node_id'])

    def set_unknown_node_for_unlinked(self) -> None:
        with BatchInserter(settings.IDM_INCONSISTENCY_BATCH_SIZE) as inserter:
            for role in self.remote_roles:
                identity = self.index_to_identity[role.identity]
                if identity not in self.identity_to_node_id:
                    inconsistency = self._create_inconsistency(role, INC_TYPE.UNKNOWN_ROLE, identity)
                    inserter.insert(inconsistency)
        # Оставляем только роли системы с найденными узлами
        self.remote_roles = [role for role in self.remote_roles if self.index_to_identity[role.identity] in self.identity_to_node_id]

        # заменяем identity_index на node_id, преобразуем массив во множество
        self.remote_roles_set = {
            RemoteRole(
                item.user, item.group, item.subject_type,
                self.identity_to_node_id[self.index_to_identity[item.identity]], item.fields,
                item.with_inheritance, item.with_robots, item.with_external,
            )
            for item in self.remote_roles
            if (self.system.group_policy in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS or item[1] is None)
        }
        del self.remote_roles

    def prepare_active_roles(self) -> None:
        # Дополнительные флаги добавляем только для "нового" плагина с поддержкой групповых ролей
        # Для систем, не знающих о группах, групповые роли не учитываем
        roles_qs = (
            self.system.roles
            .filter(Q(is_active=True) | Q(updated__gte=self.action.added))
            .annotate(NULL_TYPE=Value(None, IntegerField()))
        )
        values_list = [
            'user_id', 'group_id', 'user__type',
            'node_id', 'system_specific',
            'with_inheritance', 'with_robots', 'with_external',
            'parent_id', 'updated', 'id',
        ]

        if self.system.group_policy not in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS:
            roles_qs = roles_qs.filter(group__isnull=True)

        # Значения, которые не поддерживаются системой, не учитываются в сверке и заменяются на None
        nullable_fields = []
        if not self.system.use_tvm_role:
            nullable_fields.append('user__type')
        if not (self.system.group_policy in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS and isinstance(self.system.plugin, NewPlugin)):
            # тут мы начинаем считать, что в полях всегда None
            # из системы нам действительно может приехать только None
            # а в систему мы всё равно не сможем поля запушить, поэтому можно считать, что тут None
            nullable_fields.extend(['with_inheritance', 'with_robots', 'with_external'])
        values_list = [('NULL_TYPE' if field in nullable_fields else field) for field in values_list]
        # Получаем индексы, чтобы не завязываться на конкретный набор полей
        fields_idx = values_list.index('system_specific')
        parent_idx = values_list.index('parent_id')
        updated_idx = values_list.index('updated')
        id_idx = values_list.index('id')

        roles_list = roles_qs.values_list(*values_list)
        roles_list = [
            # Сериализуем system_specific в строку
            (
                role[:fields_idx] +
                (role[fields_idx] and json.dumps(role[fields_idx], sort_keys=True),) +
                role[fields_idx + 1:]
            )
            for role
            in roles_list
        ]
        without_parent = [role for role in roles_list if role[parent_idx] is None]
        with_parent = [role for role in roles_list if role[parent_idx] is not None]
        # роли с родителем должны перезаписывать всё остальное, т.е. они важнее, при этом сами родители нам не нужны
        # upper – те, которые или активны, или возможно недавно отозваны
        # lower – только активные
        self.active_roles_upper = {
            RemoteRole(*role[:parent_idx]): role[id_idx]
            for role in without_parent
        }
        self.active_roles_upper.update({
            RemoteRole(*role[:parent_idx]): role[id_idx]
            for role in with_parent
        })
        self.active_roles_upper_set = set(self.active_roles_upper)
        self.active_roles_lower = {
            RemoteRole(*role[:parent_idx]): role[id_idx]
            for role in without_parent
            if role[updated_idx] < self.action.added
        }
        self.active_roles_lower.update({
            RemoteRole(*role[:parent_idx]): role[id_idx]
            for role in with_parent
            if role[updated_idx] < self.action.added
        })
        self.active_roles_lower_set = set(self.active_roles_lower)

    def create_side(self, to_include: Set[RemoteRole], to_exclude: Set[RemoteRole], inconsistency_type: str) -> None:
        diff = to_include - to_exclude
        with BatchInserter(settings.IDM_INCONSISTENCY_BATCH_SIZE) as inserter:
            for role in diff:
                inconsistency = self._create_inconsistency(role, inconsistency_type)
                inserter.insert(inconsistency)

    def create_ours(self) -> None:
        self.create_side(self.active_roles_lower_set, self.remote_roles_set, INC_TYPE.OUR)

    def create_theirs(self) -> None:
        self.create_side(self.remote_roles_set, self.active_roles_upper_set, INC_TYPE.THEIR)


def transform_all_roles(method_name, plugin_roles) -> Tuple[List, str]:
    if method_name == SYSTEM_AUDIT_METHOD.GET_ROLES:
        ident_type = 'path'
        all_roles = plugin_roles
    elif method_name == SYSTEM_AUDIT_METHOD.GET_ALL_ROLES:
        ident_type = 'data'
        # все равно все потом в память поднимается, а так можно отдельно узнать время скачивания
        all_roles = list(stream_roles(plugin_roles))
    else:
        raise ValueError(method_name)

    return all_roles, ident_type


def check_roles(system, threshold, action):
    if threshold is NOT_PROVIDED:
        threshold = (
            system.inconsistencies_for_break
            if system.inconsistencies_for_break is not None
            else settings.IDM_SYSTEM_BREAKDOWN_INCONSYSTENCY_COUNT
        )

    log.info('Checking roles in {0}'.format(system.slug))

    with log_context(system=system.slug, source='audit_memory',
                     step=INCONSISTENCY_AUDIT_STEPS.GET_ROLES, unique_id=action.pk):
        start = time.monotonic()
        method_name, plugin_roles = get_roles(system)
        if plugin_roles is NotImplemented:
            return True
        elif plugin_roles is False:
            return False

        all_roles, ident_type = transform_all_roles(method_name, plugin_roles)
        with log_context(total_time=time.monotonic() - start, audit_step_finished=True):
            log.info('Finished get_roles for system %s', system.slug)

    users = {user.username: user for user in User.objects.all()}
    users_per_uid = {user.uid: user for user in users.values() if user.uid}
    groups = {}
    for group in Group.objects.user_groups():
        if group.is_active() or group.external_id not in groups:
            groups[group.external_id] = group

    # узнаем, сколько до прогона было неконсистентностей логина/группы
    unknown_subjects_count = system.inconsistencies.filter(
        state=INC_STATE.ACTIVE, type__in=(INC_TYPE.UNKNOWN_GROUP, INC_TYPE.UNKNOWN_USER)
    ).count()
    # закроем все старые неконсистентности
    system.inconsistencies.filter(state=INC_STATE.ACTIVE).update(state=INC_STATE.OBSOLETE)

    user_id_to_username = {user.id: username for (username, user) in users.items()}
    group_id_to_external_id = {group.id: external_id for (external_id, group) in groups.items()}

    remote_roles = []
    identity_to_index = {}
    index_to_identity = {}
    last_index = -1
    with BatchInserter(settings.IDM_INCONSISTENCY_BATCH_SIZE) as inserter, \
            log_context(system=system.slug, source='audit_memory',
                        step=INCONSISTENCY_AUDIT_STEPS.INSERT_INCONSISTENCIES, unique_id=action.pk):
        for role in all_roles:
            username = group_id = user = uid = group = path = data = None
            type_ = INC_TYPE.UNDECIDED_YET
            state = INC_STATE.CREATED
            if 'uid' in role:
                uid = role['uid']
                if uid in users_per_uid:
                    user = users_per_uid[uid]
                else:
                    type_ = INC_TYPE.UNKNOWN_USER
                    state = INC_STATE.ACTIVE
                    username = f'uid:{uid}'
            elif 'login' in role:
                username = role['login'].lower().strip()
                if username in users:
                    user = users[username]
                else:
                    type_ = INC_TYPE.UNKNOWN_USER
                    state = INC_STATE.ACTIVE
            elif 'group' in role:
                if system.group_policy not in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS:
                    log.warning('%s system %s audit response contains group roles', system.group_policy, system.slug)
                    continue
                group_id = int(role['group'])
                if group_id in groups:
                    group = groups[group_id]
                else:
                    type_ = INC_TYPE.UNKNOWN_GROUP
                    state = INC_STATE.ACTIVE
            else:
                keys = ','.join(map(str, role.keys()))
                log.warning('System %s responded with unknown roles type: %s', system.slug, keys)
                continue
            if ident_type == 'path':
                path = role['path']
            else:
                data = role['data']

            fields = role.get('fields') or None
            subject_type = system.use_tvm_role and role.get('subject_type') or None

            if type_ in [INC_TYPE.UNKNOWN_GROUP, INC_TYPE.UNKNOWN_USER]:
                inconsistency = Inconsistency(
                    system=system,
                    sync_key=action,
                    state=state,
                    ident_type=ident_type,
                    remote_uid=uid,
                    remote_username=username,
                    remote_group=group_id,
                    remote_path=path,
                    remote_data=data,
                    remote_fields=fields,
                    remote_subject_type=subject_type,
                    user=user,
                    group=group,
                    type=type_,
                    with_inheritance=role.get('with_inheritance'),
                    with_robots=role.get('with_robots'),
                    with_external=role.get('with_external'),
                )
                inconsistency.remote_hash = inconsistency.calculate_hash()
                inserter.insert(inconsistency)
            else:
                identity = json.dumps(data, sort_keys=True) if system.audit_method == SYSTEM_AUDIT_METHOD.GET_ALL_ROLES else path
                if identity not in identity_to_index:
                    last_index += 1
                    identity_to_index[identity] = last_index
                    index_to_identity[last_index] = identity
                    identity_index = last_index
                else:
                    identity_index = identity_to_index[identity]

                remote_roles.append(RemoteRole(
                    user and user.id,
                    group and group.id,
                    subject_type,
                    identity_index,
                    fields and json.dumps(fields, sort_keys=True),
                    role.get('with_inheritance'),
                    role.get('with_robots'),
                    role.get('with_external'),
                ))

        with log_context(total_time=time.monotonic() - start, audit_step_finished=True):
            log.info('Finished insert_inconsistencies for system %s', system.slug)

    try:
        processor = InconsistencyProcessor(
            system, action, ident_type, remote_roles,
            index_to_identity, user_id_to_username, group_id_to_external_id,
        )
        processor.process()
    except Exception as e:
        log.exception('Error while processing inconsistencies')
        raise

    active_inconsistencies = Inconsistency.objects.filter(sync_key=action, system=system, state=INC_STATE.ACTIVE)
    inconsistency_count = active_inconsistencies.count_to_break_system(system, unknown_subjects_count)
    log.info('got %d inconsistencies in system %s', inconsistency_count, system.slug)

    if (system.can_be_broken and
        system.inconsistency_policy != 'trust' and
        threshold is not None and
        inconsistency_count > threshold
    ):
        active_inconsistencies = active_inconsistencies.select_related('user', 'group', 'node')
        unknown_usernames = active_inconsistencies.filter(type=INC_TYPE.UNKNOWN_USER)
        unknown_groups = active_inconsistencies.filter(type=INC_TYPE.UNKNOWN_GROUP)
        we_have_system_dont = active_inconsistencies.filter(type=INC_TYPE.OUR)
        system_has_we_dont = active_inconsistencies.filter(type=INC_TYPE.THEIR)
        unknown_roles = active_inconsistencies.filter(type=INC_TYPE.UNKNOWN_ROLE)

        log.warning('blocking system %s', system.slug)
        we_have = defaultdict(list)
        for inconsistency in we_have_system_dont:
            we_have[inconsistency.get_owner()].append(inconsistency.node)
        system_has = defaultdict(list)
        for inconsistency in system_has_we_dont:
            system_has[inconsistency.get_owner()].append(inconsistency.node)
        system_has_unknown_roles = defaultdict(list)
        for inconsistency in unknown_roles:
            system_has_unknown_roles[inconsistency.get_owner()].append(inconsistency)
        context = {
            'we_have_system_dont': list(we_have.items()),
            'system_has_we_dont': list(system_has.items()),
            'unknown_usernames': unknown_usernames.values_list('remote_username', flat=True),
            'unknown_groups': unknown_groups.values_list('remote_group', flat=True),
            'unknown_roles': list(system_has_unknown_roles.items()),
            'total_count': inconsistency_count,
        }
        system.set_broken(
            reason=_('Слишком много расхождений: %s') % inconsistency_count,
            context=context,
        )

    return True
