# coding: utf-8


import logging

from django.conf import settings
from django.db import models
from django.db.models import Count, NOT_PROVIDED, Q, QuerySet
from django.utils.translation import ugettext_lazy as _

from idm.core.constants.action import ACTION
from idm.core.constants.system import SYSTEM_INCONSISTENCY_POLICY, SYSTEM_AUDIT_BACKEND
from idm.core.constants.groupmembership_system_relation import GROUPMEMBERSHIP_INCONSISTENCY
from idm.core.exceptions import SynchronizationError
from idm.core.models import System
from idm.inconsistencies.group_memberships import check_group_memberships
from idm.notification.utils import send_notification, create_sox_incosistency_issue
from idm.utils.actions import start_stop_actions
from idm.utils.lock import lock

log = logging.getLogger(__name__)


class MatchingRoleQuerySet(QuerySet):
    def active(self):
        return self.filter(is_active=True)


class InconsistencyQuerySet(QuerySet):
    def active(self):
        return self.filter(state='active')

    def inactive(self):
        return self.exclude(state='active')

    def our(self):
        return self.filter(type=self.model.TYPE_OUR)

    def their(self):
        return self.filter(type=self.model.TYPE_THEIR)

    def unresolvable_for_system(self, system):
        qs = self.filter(system=system)
        removed_node_query = Q(type=self.model.TYPE_OUR, node__state='depriving')
        if system.inconsistency_policy == SYSTEM_INCONSISTENCY_POLICY.STRICT_SOX:
            return qs.filter(removed_node_query)
        else:
            return qs.filter(removed_node_query | Q(type__in=self.model.DEFAULT_UNRESOLVABLE_TYPES))

    def resolvable_for_system(self, system):
        resolvable = self.model.DEFAULT_RESOLVABLE_TYPES
        if system.inconsistency_policy == SYSTEM_INCONSISTENCY_POLICY.STRICT_SOX:
            resolvable = self.model.DEFAULT_RESOLVABLE_TYPES | self.model.DEFAULT_UNRESOLVABLE_TYPES
        qs = self.filter(system=system, type__in=resolvable)
        qs = qs.exclude(type=self.model.TYPE_OUR, node__isnull=False, node__state='depriving')
        return qs.select_related('system')

    def count_to_break_system(self, system, unknown_subjects_count=None):
        active = self.filter(system=system, state='active')
        count = active.count()

        # Не учитываем случай с удалением узла со стороны системы
        count -= active.filter(type=self.model.TYPE_OUR, node__state='depriving').count()

        new_unknown_subjects_count = active.filter(
            type__in=(self.model.TYPE_UNKNOWN_GROUP, self.model.TYPE_UNKNOWN_USER)
        ).count()
        # если вновь обнаружено больше и столько же неконсистентностей логина, то не будем учитывать
        # старые неконсистентности логина при сравнении с пороговым значением
        if unknown_subjects_count is not None and new_unknown_subjects_count >= unknown_subjects_count:
            count -= unknown_subjects_count
        else:
            # если же обнаружено меньше неконсистентностей логина, чем раньше, то не учитываем
            # новые неконсистентности логина
            count -= new_unknown_subjects_count
        return count


class MatchingRoleManager(models.Manager.from_queryset(MatchingRoleQuerySet)):
    """Менеджер для MatchingRole"""


class InconsistencyManager(models.Manager.from_queryset(InconsistencyQuerySet)):
    @staticmethod
    def check_system(system: System, threshold: int = NOT_PROVIDED, requester: 'User' = None):
        result = False
        with start_stop_actions(
                ACTION.STARTED_COMPARISON_WITH_SYSTEM,
                ACTION.COMPARED_WITH_SYSTEM,
                extra={'requester': requester, 'system': system},
        ) as manager:
            manager.on_failure({'data': {'status': 1}})
            manager.on_failure(lambda mngr: log.exception('Error while checking system %s', system.slug))
            manager.on_success({'data': {'status': 0}})
            with lock('idm.inconsistencies.querysets.check_system:%s' % system.slug, block=False) as acquired:
                if not acquired:
                    raise SynchronizationError('Cannot acquire lock for checking system %s' % system.slug)
                if system.audit_backend == SYSTEM_AUDIT_BACKEND.DATABASE:
                    from idm.inconsistencies.audit import check_roles as check_roles_db
                    check_roles_func = check_roles_db
                elif system.audit_backend == SYSTEM_AUDIT_BACKEND.MEMORY:
                    from idm.inconsistencies.audit_memory import check_roles as check_roles_memory
                    check_roles_func = check_roles_memory
                else:
                    raise AssertionError('Unsupported audit_backend value: {}'.format(system.audit_backend))
                result = check_roles_func(system, threshold=threshold, action=manager.start_action)
                if result is False:
                    raise SynchronizationError('check_roles returned False')
        return result

    def check_and_resolve(self, system, threshold=NOT_PROVIDED, requester=None, force=False):
        result = self.check_system(system, threshold, requester)
        if result:
            self.resolve_system(system, threshold=threshold, force=force, requester=requester)

    def check_roles(self, system_slug=None, threshold=NOT_PROVIDED, requester=None):
        log.info('starting to check roles for system: "%s" and with threshold: %s',
                 system_slug or 'all', threshold if threshold is not NOT_PROVIDED else 'default thresholds')
        systems = System.objects.operational().nondumb()
        # если была указана конкретная система для проверки
        if system_slug:
            systems = systems.filter(slug=system_slug)
        results = []
        for system in systems:
            result = self.check_system(system, threshold, requester)
            results.append(result)
        return all(results)

    def resolve_system(self, system, threshold=NOT_PROVIDED, force=False, requester=None, resolve_in_idm_direct=False):
        if not system.is_operational():
            log.warning('Cannot resolve inconsistencies for system %s: the system is not operational', system.slug)
            return False

        inconsistencies_count = self.count_to_break_system(system)
        if threshold is NOT_PROVIDED:
            threshold = (
                system.inconsistencies_for_break
                if system.inconsistencies_for_break is not None
                else settings.IDM_SYSTEM_BREAKDOWN_INCONSYSTENCY_COUNT
            )
        if threshold is not None and inconsistencies_count > threshold and system.inconsistency_policy != 'trust':
            log.warning('Cannot resolve %d inconsistencies for system %s: too many inconsistencies',
                        inconsistencies_count, system.slug)
            return False

        with start_stop_actions(
                ACTION.STARTED_SYNC_WITH_SYSTEM,
                ACTION.SYNCED_WITH_SYSTEM,
                extra={'requester': requester, 'system': system},
        ) as manager:
            manager.on_failure({'data': {'status': 1}})
            manager.on_failure(lambda mngr: log.exception('Error while resolving system %s', system.slug))
            with lock('idm.inconsistencies.querysets.resolve_system:%s' % system.slug, block=False) as acquired:
                if not acquired:
                    raise SynchronizationError('Cannot acquire lock for resolving system %s' % system.slug)
                roles = system.roles.filter(is_active=True)
                inconsistencies = self.active()
                unresolvable = inconsistencies.unresolvable_for_system(system)
                resolvable = (
                    inconsistencies
                    .resolvable_for_system(system)
                    .select_related(
                        'system__actual_workflow', 'node__system', 'our_role__node', 'our_role__system__actual_workflow'
                    )
                )

                errors = unresolvable.count()

                log.info('Trying to resolve %d inconsistencies', resolvable.count())
                existed = roles.count()

                resolved = created_total = deprived_total = 0

                for inconsistency in resolvable:
                    try:
                        # пытаемся разрешить неконсистентность автоматически от имени робота,
                        # спрашивая согласия аппруверов при создании роли в IDM
                        created, deprived = inconsistency.resolve(
                            force=force,
                            resolve_in_idm_direct=resolve_in_idm_direct,
                        )
                        created_total += created
                        deprived_total += deprived
                        resolved += 1
                    except Exception:
                        errors += 1
                        log.exception('Error during resolve inconsistency %s', inconsistency)

                if force:
                    resolved_inconsistencies = self.filter(state='resolved', is_forced=True)
                    user_resolved_inconsistencies = resolved_inconsistencies.filter(user__is_active=True)
                    user_resolved_inconsistencies.delete()
                    group_resolved_inconsistencies = resolved_inconsistencies.filter(group__state='active')
                    group_resolved_inconsistencies.delete()
                manager.on_success({
                    'data': {
                        'status': 0,
                        'report_existed': existed,
                        'report_created': resolved,
                        'report_created_count': created_total,
                        'report_deprived_count': deprived_total,
                        'report_errors': errors
                    }
                })
        return True

    def resolve(self, system_slug=None, force=False):
        """
        Разрешает все неразрешённые неконсистентности для существующих пользователей
        и несломанных систем в пользу системы.
        :param system_slug: Слаг системы, для которой разрешаются неконсистентности.
        Если None, то разрешаем неконсистентности для всех систем.
        :param force: Принудительно разрешать зависимости, не спрашивая одобрения аппруверов.
        :return: None
        """
        systems = System.objects.get_operational()
        # если была указана конкретная система для проверки
        if system_slug:
            systems = systems.filter(slug=system_slug)
        results = []
        for system in systems:
            result = self.resolve_system(system, force=force)
            results.append(result)
        return all(results)

    def get_report(self, system, format=None):
        """Генерим данные для отправки отчета по почте."""

        assert format in (None, 'short', 'full')

        report = {}
        if format is None:
            inconsistencies = self.active().filter(system=system)
            inconsistencies_count = inconsistencies.count()
            if inconsistencies_count >= settings.IDM_INCONSISTENCY_REPORT_THRESHOLD:
                report = self.get_short_report(system)
            elif inconsistencies_count > 0:
                report = self.get_full_report(system)
        elif format == 'short':
            report = self.get_short_report(system)
        elif format == 'full':
            report = self.get_full_report(system)
        return report

    def get_short_report(self, system):
        report = {
            system: {'format': 'short'}
        }
        inconsistencies = self.active().filter(system=system)
        annotated = (
            inconsistencies.
            values('type').
            annotate(type_count=Count('type')).
            order_by('-type_count').
            values_list('type', 'type_count')
        )
        stats = {}
        types = dict(self.model.TYPES)
        for type_, count in annotated:
            stats[type_] = {
                'count': count,
                'name': types[type_]
            }
        report[system]['statistics'] = stats
        return report

    def get_full_report(self, system, types=None):
        report = {
            system: {'format': 'full'}
        }
        inconsistencies = self.active().filter(system=system)
        if types is not None:
            inconsistencies = inconsistencies.filter(type__in=types)
        if inconsistencies.count() == 0:
            return None

        for obj in inconsistencies.select_related('system', 'user', 'group'):
            if obj.type in self.model.OWNERLESS_TYPES:
                report[system].setdefault(obj.type, [])
                report[system][obj.type].append(obj)
            else:
                owner = obj.get_owner()
                report[system].setdefault(obj.type, {})
                report[system][obj.type].setdefault(owner, [])
                report[system][obj.type][owner].append(obj)
        return report

    def send_report(self, system):
        log.info('Generating inconsistency report for system %s', system.slug)
        report = self.get_report(system)
        if not report:
            # формируем отчет только в том случае, если действительно
            # обнаружена неконсистентность
            return True  # но мы точно сделали всё, что должны были (то есть ничего)

        result = False
        context = {
            'report': report,
        }
        try:
            if system.inconsistency_policy == SYSTEM_INCONSISTENCY_POLICY.STRICT_SOX:
                subreport = self.get_full_report(system, types=[self.model.TYPE_UNKNOWN_ROLE])
                if subreport:
                    create_sox_incosistency_issue(system, subreport)
                log.info('Inconsistency issue for system %s was successfully created', system.slug)
            send_notification(
                _('Нарушение консистентности в системе %s') % system.get_name(lang='ru'),
                ['emails/service/report_check_roles.txt'],
                system.get_emails(fallback_to_reponsibles=True) + list(settings.EMAILS_FOR_REPORTS),
                context
            )
            log.info('Inconsistency report for system %s was successfully sent', system.slug)
            result = True
        except Exception:
            log.exception('Error while sending inconsistency report for system %s', system.slug)
        return result

    def expire_alike(self, role):
        """Получить список type=THEIR неконсистентностей, которые аналогичны роли role, и перевести их в obsolete,
        как более нерелевантные"""
        qs = self.get_queryset().filter(
            user_id=role.user_id,
            group_id=role.group_id,
            system_id=role.system_id,
            type=self.model.TYPE_THEIR,
            state='active',
            node_id=role.node_id,
            remote_fields=role.system_specific
        )
        qs.update(state='obsolete')
        return qs


class GroupMembershipInconsistencyQuerySet(QuerySet):
    def active(self):
        return self.filter(state=GROUPMEMBERSHIP_INCONSISTENCY.STATES.ACTIVE)

    def active_our_side_inconsistencies(self):
        return self.active().filter(type=GROUPMEMBERSHIP_INCONSISTENCY.TYPES.OUR)

    def active_system_side_inconsistencies(self):
        return self.active().filter(type=GROUPMEMBERSHIP_INCONSISTENCY.TYPES.THEIR)


class GroupMembershipInconsistencyManager(models.Manager.from_queryset(GroupMembershipInconsistencyQuerySet)):
    @staticmethod
    def check_system(system: System, requester: 'User' = None):
        with start_stop_actions(
                ACTION.STARTED_MEMBERSHIP_SYNC,
                ACTION.FINISHED_MEMBERSHIP_SYNC,
                extra={'system': system, 'requester': requester},
        ) as manager:
            manager.on_failure({'data': {'status': 1}})
            manager.on_failure(lambda mngr: log.exception('Error while checking system %s', system.slug))
            manager.on_success({'data': {'status': 0}})
            with lock(
                    'idm.groupmembership_inconsistencies.querysets.check_system:%s' % system.slug,
                    block=False,
            ) as acquired:
                if not acquired:
                    raise SynchronizationError('Cannot acquire lock for checking system %s memberships' % system.slug)
                try:
                    check_group_memberships(system, action=manager.start_action)
                except:
                    raise SynchronizationError('check_memberships failed')
