# coding: utf-8


import hashlib
import logging

from django.db import models, transaction
from django.dispatch import receiver
from django.utils.encoding import force_bytes, force_text
from django.utils.translation import ugettext as _
from django_pgaas import atomic_retry

from idm.core.constants.role import ROLE_STATE
from idm.core.constants.groupmembership_system_relation import GROUPMEMBERSHIP_INCONSISTENCY
from idm.core.constants.system import SYSTEM_INCONSISTENCY_POLICY
from idm.core.workflow.exceptions import Forbidden
from idm.core.exceptions import UnresolvableInconsistencyError
from idm.core.models import Role
from idm.core.signals import role_changed_state
from idm.core.workflow.common.subject import subjectify
from idm.framework.fields import NullJSONField as JSONField
from idm.framework.utils import add_to_instance_cache
from idm.inconsistencies.managers import (
    InconsistencyManager,
    MatchingRoleManager,
    GroupMembershipInconsistencyManager,
)
from idm.utils import json

log = logging.getLogger(__name__)


class Inconsistency(models.Model):
    class TYPE:
        OUR = 'we_have_system_dont'
        THEIR = 'system_has_we_dont'
        UNKNOWN_ROLE = 'system_has_unknown_role'
        UNKNOWN_USER = 'system_has_unknown_user'
        UNKNOWN_GROUP = 'system_has_unknown_group'
        UNDECIDED_YET = 'undecided'

    class STATE:
        CREATED = 'created'
        ACTIVE = 'active'
        MATCHING = 'matching'
        RESOLVED = 'resolved'
        OBSOLETE = 'obsolete'

    TYPE_OUR = TYPE.OUR
    TYPE_THEIR = TYPE.THEIR
    TYPE_UNKNOWN_ROLE = TYPE.UNKNOWN_ROLE
    TYPE_UNKNOWN_USER = TYPE.UNKNOWN_USER
    TYPE_UNKNOWN_GROUP = TYPE.UNKNOWN_GROUP
    TYPE_UNDECIDED_YET = TYPE.UNDECIDED_YET

    ROLE_IDENT_TYPES = (
        ('data', _('По data')),
        ('path', _('По slug_path')),
    )
    STATE_CHOICES = (
        (STATE.CREATED, _('Создана')),
        (STATE.ACTIVE, _('Активна')),
        (STATE.MATCHING, _('Совпадение')),
        (STATE.RESOLVED, _('Разрешена')),
        (STATE.OBSOLETE, _('Устарела')),
    )

    TYPES = (
        (TYPE_UNDECIDED_YET, _('Тип ещё не определён')),
        (TYPE_OUR, _('У нас есть роль, в системе – нет')),
        (TYPE_THEIR, _('В системе есть роль, у нас – нет')),
        (TYPE_UNKNOWN_ROLE, _('В системе есть роль, но она отсутствует в дереве ролей системы')),
        (TYPE_UNKNOWN_USER, _('В системе имеется пользователь, о котором мы не знаем')),
        (TYPE_UNKNOWN_GROUP, _('В системе имеется группа, о которой мы не знаем'))
    )
    ALL_TYPES = {TYPE_OUR, TYPE_THEIR, TYPE_UNKNOWN_USER, TYPE_UNKNOWN_GROUP, TYPE_UNKNOWN_ROLE, TYPE_UNDECIDED_YET}
    DEFAULT_RESOLVABLE_TYPES = {TYPE_OUR, TYPE_THEIR}
    DEFAULT_UNRESOLVABLE_TYPES = {TYPE_UNKNOWN_USER, TYPE_UNKNOWN_GROUP, TYPE_UNKNOWN_ROLE}
    OWNERLESS_TYPES = {TYPE_UNKNOWN_USER, TYPE_UNKNOWN_GROUP}

    # известные заранее поля
    sync_key = models.ForeignKey('core.Action', related_name='inconsistencies', null=True, blank=False,
                                 verbose_name=_('Действие начала синхронизации'), on_delete=models.CASCADE)
    new_sync_key_id = models.BigIntegerField(null=True, blank=True, serialize=False)
    system = models.ForeignKey('core.System', related_name='inconsistencies', null=True, blank=False,
                               db_constraint=False, verbose_name=_('Система'), on_delete=models.CASCADE)

    # состояние
    state = models.CharField(max_length=50, choices=STATE_CHOICES, default='', db_index=True,
                             verbose_name=_('Состояние расхождения'))
    is_forced = models.BooleanField(default=False, verbose_name=_('Расхождение разрешено без подтверждения'))

    # данные, полученные от системы плюс флаги, вычисленные сразу по этим данным
    ident_type = models.CharField(max_length=50, choices=ROLE_IDENT_TYPES, default='data', db_index=True,
                                  verbose_name=_('Способ идентификации роли'))
    remote_username = models.CharField(max_length=30, null=True, blank=True, db_index=True,
                                       verbose_name=_('Переданный username'))
    remote_uid = models.CharField(max_length=64, null=True, blank=True, db_index=True, verbose_name=_('Переданный uid'))
    remote_group = models.IntegerField(null=True, blank=True, db_index=True, verbose_name=_('Переданный ID группы'))
    remote_path = models.TextField(null=True, blank=True, db_index=True, verbose_name=_('Переданный path'))
    remote_data = JSONField(null=True, blank=True, db_index=True, verbose_name=_('Переданные data'))

    remote_fields = JSONField(null=True, blank=True, db_index=True, verbose_name=_('Переданные поля'))

    remote_hash = models.CharField(null=True, blank=True, max_length=40,
                                   verbose_name=_('Хеш от значений data/path и fields_data'))
    remote_subject_type = models.TextField(null=True, blank=True, verbose_name=_('Значение subject_type'))

    # вычисленные по пришедшим данным поля
    user = models.ForeignKey('users.User', related_name='inconsistencies', null=True, blank=True,
                             db_constraint=False, verbose_name=_('Сотрудник'), on_delete=models.CASCADE)
    group = models.ForeignKey('users.Group', related_name='inconsistencies', null=True, blank=True,
                              db_constraint=False, verbose_name=_('Группа'), on_delete=models.CASCADE)
    node = models.ForeignKey('core.RoleNode', related_name='inconsistencies', null=True, blank=True,
                             db_constraint=False, verbose_name=_('Узел дерева ролей'), on_delete=models.CASCADE)
    our_role = models.ForeignKey('core.Role', null=True, blank=True, related_name='inconsistencies',
                                 db_constraint=False, verbose_name=_('Роль, которая есть у нас'), on_delete=models.CASCADE)
    type = models.CharField(max_length=50, choices=TYPES, default='', db_index=True,
                            verbose_name=_('Тип расхождения'))
    added = models.DateTimeField(auto_now_add=True, editable=False)
    updated = models.DateTimeField(auto_now=True, editable=False)
    with_inheritance = models.NullBooleanField(verbose_name=_('Будет ли роль выдана вложенным департаментам'))
    with_external = models.NullBooleanField(verbose_name=_('Будет ли роль выдана внешним'))
    with_robots = models.NullBooleanField(verbose_name=_('Будет ли роль выдана роботам'))

    objects = InconsistencyManager()

    class Meta:
        db_table = 'upravlyator_inconsistency'
        app_label = 'core'  # влияет на выдаваемые permissions, важно
        verbose_name = _('Расхождение')
        verbose_name_plural = _('Расхождения')
        unique_together = (('sync_key', 'state', 'type', 'remote_hash'),)
        index_together = (
            ('node', 'user', 'remote_fields'),
            ('node', 'group', 'remote_fields'),
            ('system', 'sync_key', 'state', 'node'),
            ('remote_data', 'ident_type', 'system', 'sync_key', 'state'),
            ('remote_path', 'ident_type', 'system', 'sync_key', 'state'),
        )

    def __str__(self):
        if self.user or self.group:
            subject_name = self.get_subject().get_ident()
        else:
            subject_name = '(None)'
        result = 'Inconsistency [id:%(id)d, type:%(type)s] in system %(system)s for %(subject)s. State: %(state)s' % {
            'id': self.pk,
            'type': self.type,
            'system': self.system.slug,
            'subject': subject_name,
            'state': self.state
        }
        return result

    def calculate_hash(self):
        # На этапе вызова этой функции нельзя использовать is_resolvable, так как тип ещё точно не определён,
        # единственное что мы знаем – есть владелец или нет. Но этого достаточно:
        # в sox-системе мы всегда используем узел и fields_data в хеше, а в не-sox – во всех случаях, кроме случая,
        # когда владелец нам неизвестен.
        if self.system.inconsistency_policy == 'donottrust_strict':
            use_roledata = True
        else:
            use_roledata = self.type not in self.OWNERLESS_TYPES
        if self.remote_uid is not None:
            owner_source = self.remote_uid
        elif self.remote_username is not None:
            owner_source = self.remote_username
        else:
            owner_source = str(self.remote_group)
        if self.remote_path is not None:
            node_source = self.remote_path
        else:
            node_source = json.dumps(self.remote_data, sort_keys=True)
        hasher = hashlib.sha1(force_bytes(owner_source))
        if use_roledata:
            hasher.update(force_bytes(node_source))
        if self.remote_fields is not None and use_roledata:
            hasher.update(force_bytes(json.dumps(self.remote_fields)))
        return hasher.hexdigest()

    def is_resolvable(self):
        resolvable = self.DEFAULT_RESOLVABLE_TYPES
        if self.system.inconsistency_policy == 'donottrust_strict':
            resolvable = self.DEFAULT_RESOLVABLE_TYPES | self.DEFAULT_UNRESOLVABLE_TYPES
        return self.type in resolvable

    def get_ident(self):
        if self.type == self.TYPE_UNKNOWN_USER:
            ident = self.remote_username
        elif self.type == self.TYPE_UNKNOWN_GROUP:
            ident = self.remote_group
        else:
            ident = self.get_subject().get_ident()
        return ident

    def get_owner(self):
        if self.type in self.OWNERLESS_TYPES:
            raise ValueError('Cannot subjectify unresolvable inconsistency')
        if self.user is not None:
            owner = self.user
        else:
            owner = self.group
        return owner

    def get_subject(self):
        if self.type in Inconsistency.OWNERLESS_TYPES:
            raise ValueError('Cannot subjectify unresolvable inconsistency')
        if self.user is not None:
            subject = subjectify(self.user)
        else:
            subject = subjectify(self.group)
        return subject

    def get_node_ident(self, stringify=True):
        if self.type in (self.TYPE_OUR, self.TYPE_THEIR):
            if self.node is None:
                return 'Node is not defined'
            if self.system.audit_method == 'get_roles':
                ident = self.node.slug_path
            else:
                ident = self.node.data
                if stringify:
                    ident = json.dumps(ident)
        elif self.ident_type == 'data':
            ident = self.remote_data
            if stringify:
                ident = json.dumps(ident)
        else:
            ident = self.remote_path
        return ident

    def has_fields(self):
        result = False
        if self.type == self.TYPE_OUR and self.our_role.fields_data:
            result = True
        elif self.remote_fields is not None:
            result = True
        return result

    def get_fields(self):
        if self.type == self.TYPE_OUR:
            fields = self.our_role.fields_data
        else:
            fields = self.remote_fields
        return json.dumps(fields)

    def as_report(self):
        """Представление для отчёта. Для типов TYPE_UNKNOWN - username/group_id, для TYPE_OUR/TYPE_THEIR -
        человеческое представление роли"""
        if self.type == self.TYPE_UNKNOWN_USER:
            result = self.remote_username
        elif self.type == self.TYPE_UNKNOWN_GROUP:
            result = self.remote_group
        elif self.type == self.TYPE_UNKNOWN_ROLE:
            result = 'node=%s' % self.get_node_ident()
            if self.remote_fields:
                result += ', fields=%s' % self.get_fields()
        else:
            node_ident = self.get_node_ident()
            if self.has_fields():
                node_ident += ', fields=%s' % self.get_fields()
            humanize = self.node.humanize() if self.node is not None else ''
            result = '%s (%s)' % (humanize, node_ident)
        return result

    # method has no non-db side-effect
    @atomic_retry
    def set_resolved(self, requester=None, comment=None, role=None, action_data=None):
        """
        Помечает неконсистентность как разрешенную и создает экшен для этого
        """
        self.state = 'resolved'
        self.is_resolved = True  # TODO: remove
        self.save()
        log.info('updated inconsistency %d: %s as resolved', self.id, str(self))
        if action_data is None:
            action_data = {}
        action_data['comment'] = comment or _('Разрешено расхождение')
        if self.user_id is not None or self.type == self.TYPE_UNKNOWN_USER:
            action_data['user'] = self.get_ident()
        elif self.group_id is not None or self.type == self.TYPE_UNKNOWN_GROUP:
            action_data['group'] = self.get_ident()
        self.actions.create(
            action='resolve_inconsistency',
            requester=requester,
            data=action_data,
            role=role,
            system=self.system,
            user=self.user,
            group=self.group,
            role_node=self.node,
        )

    def resolve(self, requester=None, force=False, resolve_in_idm_direct=False):
        """
        Разрешить несоответствие, возможно не спрашивая подтверждения апруверов при добавлении роли

        :type requester: User
        :param requester: пользователь, который разрешает неконсистентность.
        По дефолту - None, в отчете будет указан робот
        :type force: bool
        :param force: флаг, спрашивать ли аппруверов при добавлении роли в IDM. По дефолту - False, что значит,
        что аппруверов нужно спросить.
        :param resolve_in_idm_direct: флаг в какую сторону разрешать расхождения
        """
        if requester is not None and not requester.has_perm('core.resolve_inconsistencies'):
            raise Forbidden(_('У пользователя недостаточно прав для разрешения этого расхождения'))
        if self.state != 'active':
            log.warning(
                '%s tried to resolve already resolved inconsistency %d',
                requester.username if requester else 'robot-idm',
                self.id,
            )
            raise UnresolvableInconsistencyError('already-resolved')

        if self.type not in Inconsistency.ALL_TYPES:
            log.warning('Inconsistency %d: cannot be resolved: it is of unknown type', self.pk)
            raise UnresolvableInconsistencyError('Cannot be resolved: is of unknown type')
        elif not self.is_resolvable():
            log.warning('Inconsistency %d: cannot be resolved: it is of unresolvable type', self.pk)
            raise UnresolvableInconsistencyError('Cannot be resolved: is of unresolvable type')

        log.info('trying to resolve inconsistency %d: %s', self.id, force_text(self))
        action_data = {}
        if force:
            action_data['is_forcibly_resolved'] = True
        if resolve_in_idm_direct:
            action_data['resolve_in_idm_direct'] = True
        created = deprived = 0
        if self.type == Inconsistency.TYPE_OUR:
            created, deprived = self.resolve_ours(requester, action_data)
        elif self.type == Inconsistency.TYPE_THEIR:
            created, deprived = self.resolve_theirs(requester, action_data)
        elif self.type in self.DEFAULT_UNRESOLVABLE_TYPES:
            created, deprived = self.resolve_nonexistent_role_async(requester, action_data)
        return created, deprived

    # method has idempotent non-db side-effects
    @atomic_retry
    def resolve_ours(self, requester, action_data):
        """
        Пытаемся разрешить несоответствие, когда в системе роли нет - а у нас есть.
        """
        created, deprived = 0, 0
        comment = None
        role = None

        self.our_role.lock(retry=True)

        resolve_in_idm_direct = (
            action_data.get('resolve_in_idm_direct')
            or self.our_role.need_retry_push_to_system(self.system.inconsistency_policy)
        )

        add_to_instance_cache(self.our_role, 'system', self.system)
        if self.our_role.state in ROLE_STATE.ALREADY_INACTIVE_STATES:
            comment = _('Расхождение закрыто, так как аналогичная роль отозвана')
        elif resolve_in_idm_direct:
            self._resolve_ours_in_idm_direct(self.our_role)
        else:
            self._resolve_ours_in_system_direct(self.our_role)
            deprived = 1
            role = self.our_role

        # Разрешить неконсистентность нужно в любом случае
        self.set_resolved(requester, comment, action_data=action_data, role=role)
        return created, deprived

    def _resolve_ours_in_idm_direct(self, our_role):
        our_role.set_state(
            'approved',
            inconsistency=self,
            comment=_('Роль повторно добавлена в систему при разрешении расхождения'),
            ignore_parent_check=True,
            # Письмо/смс про такую роль отправлять не нужно
            action_data={'dont_send_notofication_with_results': True}
        )

    def _resolve_ours_in_system_direct(self, our_role):
        # статус выставляется сразу "deprived", т.к. не надо удалять роль из системы - ее там и так нет
        log.info('trying to set role %s state to "deprived" for inconsistency %d',
                 self.our_role.node.slug_path, self.pk)
        our_role.set_state('deprived', comment=_('Роль удалена при разрешении расхождения'),
                           from_any_state=True, inconsistency=self)

    # method has non-db side effects
    @transaction.atomic
    def resolve_theirs(self, requester, action_data):
        """Пытаемся разрешить несоответствие, когда в системе роль есть - а у нас нет"""
        resolve_in_idm_direct = (
                action_data.get('resolve_in_idm_direct')
                or self.system.inconsistency_policy == SYSTEM_INCONSISTENCY_POLICY.TRUST_IDM
        )
        if resolve_in_idm_direct:
            return self._resolve_theirs_in_idm_direct(requester, action_data)
        else:
            return self._resolve_theirs_in_system_direct(requester, action_data)

    def _resolve_theirs_in_system_direct(self, requester, action_data):
        """
        Добавим роль к нам для активных пользователей, толкнем плагин для отзыва роли из системы для уволенных

        :type force: bool
        :param force: флаг, спрашивать ли аппруверов при добавлении роли в IDM. По дефолту - False, что значит,
        что нужно спрашивать
        """

        created, deprived = 0, 0
        force = action_data.get('is_forcibly_resolved', False)

        if force:
            self.is_forced = True
            self.save(update_fields=('is_forced',))

            action_data['comment'] = _('Запрошена при принудительном разрешении некосистентности: '
                                       'синхронизации с системой или первичном импорте')
        else:
            action_data['comment'] = _('Запрошена при разрешении расхождения')

        subject = self.get_subject()
        if subject.is_active():
            if not self.node.is_public or self.system.inconsistency_policy == 'trust':
                # в отношении систем, которым мы доверяем, и невидимых узлов всех систем безоговорочно доверяем
                # системе и автоматически подтверждаем роль
                force = True

            # Проверяем, нет ли такой роли в статусе sent для систем, где имеет смысл такая проверка
            if self.system.role_grant_policy == 'system':
                qs = Role.objects.filter(
                    system=self.system,
                    node=self.node,
                    state='sent',
                    system_specific=self.remote_fields,
                    **subject.get_role_kwargs()
                )
                if qs.exists():
                    log.info(
                        'Found role "%s" in "sent" state in system "%s" for "%s". Try to grant it',
                        self.node.slug_path, self.system.slug, subject.get_ident()
                    )
                    existed_role = qs.get()
                    add_to_instance_cache(existed_role, 'node', self.node)
                    add_to_instance_cache(existed_role, 'system', self.system)
                    existed_role.set_state('granted', inconsistency=self)
                    existed_role.send_email_with_results(None)
                    return True, {'created_role': existed_role}

            # cоздаем роль, которая уже есть в системе со статусом imported
            log.info(
                'Create role %s for %s (%s) in system %s, system_specific %s from inconsistency',
                self.node.slug_path, subject.get_ident(), subject.get_ident(), self.system.slug, self.remote_fields
            )

            system_fields = set(field.slug for field in self.node.get_fields())
            fields_data = None
            if self.remote_fields is not None:
                fields_data = {k: v for k, v in self.remote_fields.items() if k in system_fields} or None

            if self.system.inconsistency_policy == 'donottrust_strict' and not force:
                # Неконсистентность разрешится после выполнения задачи, она вызовет для неё set_resolved
                self.system.remove_not_existing_role_and_request_new_async(self, action_data=action_data)
                deprived = 1
                created = 1
            else:
                role = Role.objects.create_role(
                    subject=subject,
                    system=self.system,
                    node=self.node,
                    fields_data=fields_data,
                    system_specific=self.remote_fields,
                    inconsistency=None if force else self,  # если force, всё равно скоро удалим неконсистентность
                    save=False
                )
                with transaction.atomic():
                    role.expire_alike()
                    role.save()
                    role.set_state('imported', inconsistency=self, force=force, action_data=action_data)

                self.set_resolved(requester, comment=None, action_data=action_data, role=role)
                log.info('role %s in state "imported" created for inconsistency %d: %s', role, self.id, str(self))
                created = 1
        else:
            deprived = 1
            # Уберем роль на стороне системы, раз пользователь уже уволен/группа уже удалена.
            # Неконсистентность разрешится после выполнения задачи, она вызовет для неё set_resolved
            self.system.remove_not_existing_role_async(self, action_data=action_data, requester=requester)
        return created, deprived

    def _resolve_theirs_in_idm_direct(self, requester, action_data):
        action_data.update({
            'resolved_because_role_unknown': True,
            'comment': _('Расхождение разрешено, так как роль отсутствует в IDM')
        })
        self.system.remove_not_existing_role_async(self, requester=requester, action_data=action_data)
        created, deprived = 0, 1
        return created, deprived

    # method has non-db side effects
    @transaction.atomic
    def resolve_nonexistent_role_async(self, requester, action_data):
        action_data.update({
            'resolved_because_role_unknown': True,
            'comment': _('Расхождение разрешено, так как узла не существует')
        })

        self.system.remove_not_existing_role_async(self, requester=requester, action_data=action_data)

        created, deprived = 0, 1
        return created, deprived

    def resolve_nonexistent_role(self, requester=None, action_data=None):
        if action_data is None:
            action_data = {}

        params = {}
        subject = None
        sox_comment = ('Removing from remote system inconsistency %(inconsistency)d of '
                       'owner %(owner)s due to strict inconsistency policy')
        sox_action_data = {
            'resolved_because_of_policy': True,
            'comment': _('Расхождение разрешено без импорта '
                         'роли, так как этого требует политика системы'),
        }

        if self.type in self.OWNERLESS_TYPES:
            if self.type == self.TYPE_UNKNOWN_USER:
                ident = self.remote_username
                params['username'] = ident
                comment = 'Removing ownerless inconsistency %(inconsistency)d for unknown username %(owner)s'
                action_data.update({
                    'comment': _('Расхождение разрешено, так как владелец неизвестен'),
                })
            else:
                ident = self.remote_group
                params['group_id'] = ident
                comment = 'Removing ownerless inconsistency %(inconsistency)d for unknown group_id %(owner)s'
        else:
            subject = self.get_subject()
            is_active = subject.is_active()
            ident = self.get_subject().get_ident()
            if subject.is_user:
                params['username'] = ident
            else:
                params['group_id'] = ident
            if self.node_id is None:
                comment = 'Removing from remote system inconsistency %(inconsistency)d for user %(owner)s'
                if not is_active:
                    comment += ' and owner was dismissed'
                action_data.update({
                    'comment': _('Расхождение разрешено, так как узла не существует в дереве ролей системы')
                })
            elif subject.is_user:
                if is_active:
                    comment = sox_comment
                    action_data.update(sox_action_data)
                else:
                    comment = 'Removing from remote system inconsistency %(inconsistency)d for dismissed user %(owner)s'
                    action_data.update({
                        'resolved_because_user_is_not_active': True,
                        'comment': _('Расхождение разрешено, так как пользователь уволен'),
                    })
            else:
                if is_active:
                    comment = sox_comment
                    action_data.update(sox_action_data)
                else:
                    comment = 'Removing from remote system inconsistency %(inconsistency)d for deleted group %(owner)s'
                    action_data.update({
                        'comment': _('Расхождение разрешено, так как группа удалена'),
                        'resolved_because_group_is_not_active': True,
                    })

        if self.node_id:
            path = self.node.slug_path
            role_data = self.node.data
        else:
            path = self.remote_path
            role_data = self.remote_data

        log.info(comment, {'inconsistency': self.pk, 'owner': ident})

        subject_kwargs = {}
        if subject is not None:
            subject_kwargs = subject.get_role_kwargs()

        action = self.actions.create(
            action='remote_remove',
            requester=requester,
            role_node=self.node,
            system=self.system,
            data=action_data,
            **subject_kwargs
        )

        params.update({
            'path': path,
            'role_data': role_data,
            'fields_data': self.remote_fields,
            'system_specific': self.remote_fields,
            'is_fired': subject is not None and not subject.is_active(),
            'subject_type': self.remote_subject_type,
            'request_id': action.id,
            'unique_id': self.node.unique_id if self.node else '',
        })
        self.system.plugin.remove_role(**params)


class MatchingRole(models.Model):
    inconsistency = models.ForeignKey(
        Inconsistency, null=True, blank=True, db_constraint=False, db_index=True, on_delete=models.CASCADE,
    )
    role = models.ForeignKey(
        'core.Role', null=True, blank=True, db_constraint=False, db_index=True, on_delete=models.CASCADE,
    )
    objects = MatchingRoleManager()

    class Meta:
        verbose_name = _('Совпадающая роль')
        verbose_name_plural = _('Совпадающие роли')
        app_label = 'core'
        db_table = 'upravlyator_matching_role'

    def __str__(self):
        return _('Совпадающая с расхождением %(inconsistency_id)d роль %(role_id)d.') % {
            'inconsistency_id': self.inconsistency.pk,
            'role_id': self.role.pk
        }


@receiver(role_changed_state)
def close_inconsistencies_on_role_grant_or_deprive(sender, role, from_state, to_state, **kwargs):
    """При выдаче роли переводим активные неконсистентности вида type=THEIR, соответствующие ей, в obsolete"""
    if to_state in ('granted', 'deprived') and role.is_pushable():
        Inconsistency.objects.expire_alike(role)


class GroupMembershipInconsistency(models.Model):

    system = models.ForeignKey('core.System', related_name='groupmembership_inconsistencies',
                               null=True, blank=False, verbose_name=_('Система'), on_delete=models.CASCADE,)
    sync_key = models.ForeignKey('core.Action', related_name='groupmembership_inconsistencies', null=True, blank=False,
                                 verbose_name=_('Действие начала синхронизации'), on_delete=models.CASCADE,)
    new_sync_key_id = models.BigIntegerField(null=True, blank=True, serialize=False)
    type = models.CharField(max_length=50, choices=GROUPMEMBERSHIP_INCONSISTENCY.TYPES, default='',
                            verbose_name=_('Тип расхождения'))
    state = models.CharField(max_length=50, choices=GROUPMEMBERSHIP_INCONSISTENCY.STATES, default='', db_index=True,
                             verbose_name=_('Состояние расхождения'))
    added = models.DateTimeField(auto_now_add=True, editable=False)
    updated = models.DateTimeField(auto_now=True, editable=False)

    membership = models.ForeignKey('core.GroupMembershipSystemRelation', related_name='inconsistencies',
                                   null=True, blank=True, verbose_name=_('Системочленство'), on_delete=models.CASCADE,)

    username = models.TextField(null=True, blank=True, db_index=True,
                                verbose_name=_('Итоговый username'))
    group_id = models.IntegerField(null=True, blank=True, db_index=True, verbose_name=_('Итоговый ID группы'))

    passport_login = models.TextField(null=True, blank=True, db_index=True,
                                      verbose_name=_('Паспортный логин'))

    objects = GroupMembershipInconsistencyManager()

    class Meta:
        db_table = 'groupmembership_inconsistency'
        app_label = 'core'
        verbose_name = _('Расхождение членств')
        verbose_name_plural = _('Расхождения членств')
        unique_together = ('sync_key', 'username', 'group_id', 'passport_login')
        index_together = ('sync_key', 'system')
