from collections import defaultdict
from typing import List, Tuple, DefaultDict, Iterator

from metrics_framework.decorators import metric

from idm.core.models import System


class LightNode:
    def __init__(self, id_: int, slug: str, slug_path: str):
        self.id = id_
        self.slug = slug
        self.slug_path = slug_path


def make_tree(system: System) -> Tuple[LightNode, DefaultDict[int, List[LightNode]]]:
    root = None
    nodes = defaultdict(list)
    for node in system.nodes.active().values_list('parent_id', 'id', 'slug', 'slug_path'):
        parent_id = node[0]
        light_node = LightNode(*node[1:])
        if parent_id is None:
            if root is not None:
                raise ValueError('Tree has more than one root')
            else:
                root = light_node
        else:
            nodes[parent_id].append(light_node)

    if root is None:
        raise ValueError('Tree has no root')

    return root, nodes


def get_correct_slug_path(parent_slug_path: str, slug: str) -> str:
    return f'{parent_slug_path}{slug}/' if parent_slug_path != '' else f'/{slug}/'


def iter_all_children(node: LightNode, tree: DefaultDict[int, List[LightNode]]) -> Iterator[LightNode]:
    yield node
    for child in tree[node.id]:
        yield from iter_all_children(child, tree)


def iter_bad_children(node: LightNode, tree: DefaultDict[int, List[LightNode]]) -> Iterator[LightNode]:
    for child in tree[node.id]:
        if child.slug_path != get_correct_slug_path(node.slug_path, child.slug):
            yield from iter_all_children(child, tree)
        else:
            yield from iter_bad_children(child, tree)


def count_wrong_slug_paths(system: System):
    root, tree = make_tree(system)
    return len(list(iter_bad_children(root, tree)))


@metric('abc_slug_paths')
def count_wrong_abc_slug_paths():
    return [{
        'slug': 'wrong_nodes_count',
        'value': count_wrong_slug_paths(System.objects.get(slug='abc')),
    }]
