# coding: utf-8
from collections import defaultdict, Counter

from django.conf import settings
from django.db.models import Count, Q

from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.models import ApproveRequest
from idm.users.models import Group, GroupMembership, GroupResponsibility
import idm.users.ranks as ranks_constants


class Category(object):
    SMALL = 'Small'
    MEDIUM = 'Medium'
    LARGE = 'Large'

    @staticmethod
    def get_category(user_count):
        if user_count < settings.METRICS_SMALL_CATEGORY_UPPER_LIMIT:
            return Category.SMALL
        elif user_count < settings.METRICS_MEDIUM_CATEGORY_UPPER_LIMIT:
            return Category.MEDIUM
        else:
            return Category.LARGE


def calc_all_users_for_head(group_id, heads, group_children, head_by_group, user_total_count_by_group,
                            user_total_count_by_head=None):
    """
    Рекурсивный подсчет всех подчиненных для каждого руководителя
    :param group_id:
    :param heads: множество руководителей для которых результат уже посчитан на пути от корневой группы до текущей
    :param group_children: список дочерних групп для каждой группы
    :param head_by_group: mapping group -> head
    :param user_total_count_by_head -> количество всех подчиненных  у руководителя
    :param user_total_count_by_group -> количество работников в этой группе и дочерних группах
    :return: количество всех подчиненных  у руководителя
    """

    if user_total_count_by_head is None:
        user_total_count_by_head = defaultdict(int)

    head = None
    added_head = False
    # если у текущей группы есть руководитель
    if group_id in head_by_group:
        head = head_by_group[group_id]
        # если количество всех подчиненных для текущего руководителя еще не посчитано
        if head not in heads:
            added_head = True
            heads.add(head)
            user_total_count_by_head[head] += user_total_count_by_group[group_id]

    for child_group_id in group_children[group_id]:
        calc_all_users_for_head(child_group_id, heads, group_children, head_by_group,
                                user_total_count_by_group, user_total_count_by_head)

    if head and added_head:
        # удалить руководителя из множества, чтобы посчитать подчиненных уже в других ветках дерева
        # руководитель удаляется в той группе, в которой был добавлен
        heads.remove(head)

    return user_total_count_by_head


def get_user_count_by_head(group_type='department'):
    """
    Считает количество всех подчиненных для каждого руководителя
    :param group_type один из ('department', 'wiki', 'service')
    :return: mapping head -> total employees count
    """
    user_total_count_by_group = defaultdict(int)
    user_count_by_group = defaultdict(int)
    group_children = defaultdict(list)
    subtract_head = {}

    root_department_group = Group.objects.get_root(type_='department')
    users_and_groups = (GroupMembership.objects.filter(
        state__in=GROUPMEMBERSHIP_STATE.ACTIVE_STATES,
        group__type='department',
    ).values('group_id', 'user__username'))
    ordered_groups = Group.objects.filter(type='department').order_by('-level').values('id', 'parent_id')
    head_by_group = dict(GroupResponsibility.objects.filter(
        group__type=group_type,
        rank=ranks_constants.HEAD, is_active=True).
        values_list('group_id', 'user__username'))

    # считает количество сотрудников в группе
    for dct in users_and_groups:
        user = dct['user__username']
        group_id = dct['group_id']
        user_count_by_group[group_id] += 1
        # если сотрудник является руководителем группы, то нужно отнять 1,
        # чтобы не считать его в качестве подчиненного у самого себя
        if head_by_group.get(group_id) == user:
            subtract_head[group_id] = True

    for group_data in ordered_groups:
        group_children[group_data['parent_id']].append(group_data['id'])

    # цикл идет снизу вверх по дереву и считает общее количество работников в каждой группе
    for group in ordered_groups:
        group_id = group['id']
        parent_id = group['parent_id']

        # если в группе кто-то работает
        if group_id in user_count_by_group:
            user_total_count_by_group[group_id] += user_count_by_group[group_id]
            if parent_id > 0:
                user_total_count_by_group[parent_id] += user_total_count_by_group[group_id]
            # не считать у руководителей самих себя в качестве подчиненных
            if group_id in subtract_head:
                user_total_count_by_group[group_id] -= 1

    user_total_count_by_head = calc_all_users_for_head(root_department_group.id, set(), group_children, head_by_group,
                                                       user_total_count_by_group)

    return user_total_count_by_head


def count_approverequests_for_categorized_heads(from_datetime, to_datetime):
    """
    Разделяет всех руководителей по количеству подчиненных на 3 группы:
        - Small: < 15
        - Medium: 15-100
        - Large: > 100

    и считает количество суммарных подтверждений для каждой группы

    :param from_datetime: для фильтрации подтверждений по переданной дате
    :param to_datetime:  для фильтрации подтверждений по переданной дате
    :return: количество суммарных подтверждений для каждой группы
    """

    result = defaultdict(int)
    date_filter = Q(added__gte=from_datetime, added__lte=to_datetime)
    approverequest_by_user = (Counter(ApproveRequest.objects.filter(date_filter)
                                      .values_list('approver__username', flat=True)))
    user_count_by_head = get_user_count_by_head()

    for head, user_count in user_count_by_head.items():
        category = Category.get_category(user_count)
        result[category] += approverequest_by_user[head]

    return result


def get_total_approverequests_count_for_top_approvers(from_datetime, to_datetime, top_size=10):
    """
    Считает и возвращает суммарное количество approverequest для топ людей с самым большим количеством подтверждений
    """
    working_appovers = (Q(approver__is_active=True) & Q(added__gte=from_datetime) & Q(added__lte=to_datetime) &
                        ~(Q(approver__username__startswith='zomb-') | Q(approver__username__startswith='robot-')))
    approvers_counter = (ApproveRequest.objects.filter(working_appovers).values_list('approver__username', flat=True)
                         .annotate(amount=Count('approver__username')).order_by('-amount').
                         values_list('approver__username', 'amount')[:top_size])

    total_approverequests_count = sum(amount for approver, amount in approvers_counter)

    return total_approverequests_count
