from django.db.models import DurationField, ExpressionWrapper, F
from django.db.models.functions import Coalesce
from django.utils import timezone
from metrics_framework.decorators import metric

from idm.users.models import User


@metric('blocking_time')
def blocking_time():
    percentiles = [90, 95, 99, 100]
    now = timezone.now()
    report_date = now - timezone.timedelta(days=30)

    time_to_block = (
        User.objects.users()
            .annotate(ldap_blocked_time=Coalesce('ldap_blocked_timestamp', timezone.now()))
            .annotate(
                blocking_duration=ExpressionWrapper(
                    F('ldap_blocked_time') - F('idm_found_out_dismissal'),
                    output_field=DurationField())
            )
            .filter(
                is_active=False,
                ldap_blocked_time__gt=report_date,
            )
            .values_list('blocking_duration', flat=True)
    )

    if not time_to_block:
        return [{'slug': f'blocking_time_{p}', 'value': 0.0} for p in percentiles]

    result = []

    for percentile in percentiles:
        index = int(len(time_to_block) / 100 * percentile)
        if index != 0:
            index = index - 1
        # В отчет отдаем время в часах
        time_in_hours = round(sorted(time_to_block)[index].total_seconds() / 60 / 60, 2)
        result.append({'slug': f'blocking_time_{percentile}', 'value': time_in_hours})
    return result
