# coding: utf-8


from collections import defaultdict
from itertools import chain

from django.utils import timezone
from django.db.models import Q

from metrics_framework.decorators import metric

from idm.core.constants.action import ACTION
from idm.core.models import Action


def get_timedelta(actions, start_predictor):
    started = False
    start_date = None
    for action in sorted(actions, key=lambda x: x['added']):
        if start_predictor(action):
            start_date = action['added']
            started = True
        elif started:
            yield (action['added'] - start_date).total_seconds()
            started = False
            start_date = None


def get_timings(start_actions, finish_actions, start_query=None, finish_query=None, start_predictor=None):
    end = timezone.now().replace(minute=0, hour=0, second=0, microsecond=0)  # Получим начало текущего дня
    begin = end - timezone.timedelta(days=1)  # и начало прошлого дня
    if not start_predictor:
        start_predictor = lambda action: action['action'] in start_actions
    finish_qs = Action.objects.filter(added__range=[begin, end], action__in=finish_actions)
    if finish_query:
        finish_qs = finish_qs.filter(finish_query)
    finishes = list(
        finish_qs
        .values('added', 'role_id', 'action', 'role__system__slug', 'data')
        .order_by()
    )

    # Считаем, что роль не могла висеть в промежуточном статусе  больше 7 дней
    # выберем данные о стартовых действиях
    start_qs = Action.objects.filter(
        added__range=[end - timezone.timedelta(days=7), end],
        action__in=start_actions,
        role_id__in={action['role_id'] for action in finishes},
    )
    if start_query:
        start_qs = start_qs.filter(start_query)
    starts = list(
        start_qs
        .values('added', 'role_id', 'action', 'role__system__slug', 'data')
        .order_by()
    )

    grouped_actions = defaultdict(lambda: defaultdict(list))

    for action in chain(starts, finishes):
        grouped_actions[action['role__system__slug']][action['role_id']].append(action)
        grouped_actions['_all_systems_'][action['role_id']].append(action)

    time_to_final_status = dict()

    for system in grouped_actions:
        timings = []
        for role in grouped_actions[system]:
            timings.extend(get_timedelta(grouped_actions[system][role], start_predictor))
        if not timings:
            continue
        timings.sort()
        time_to_final_status[system] = {
            'time_of_granted_90': timings[int(len(timings) * 0.90)],
            'time_of_granted_95': timings[int(len(timings) * 0.95)],
            'time_of_granted_99': timings[int(len(timings) * 0.99)],
        }

    return time_to_final_status, len(finishes)


def get_system_formatted_timings(start_actions, finish_actions, **kwargs):
    timings, _ = get_timings(start_actions, finish_actions, **kwargs)
    measures = []
    for system_slug, stats in timings.items():
        measure = {
            'context': {
                'system': system_slug,
            },
            'values': [{'slug': measure_name, 'value': value} for measure_name, value in stats.items()]
        }
        measures.append(measure)
    return measures


@metric('granted_roles')
def compute_granted_roles():
    stats, count = get_timings([ACTION.APPROVE], [ACTION.GRANT, ACTION.FAIL])
    return [
        {'slug': 'roles_count',        'value': count},
        {'slug': 'time_of_granted_90', 'value': stats['_all_systems_']['time_of_granted_90']},
        {'slug': 'time_of_granted_95', 'value': stats['_all_systems_']['time_of_granted_95']},
        {'slug': 'time_of_granted_99', 'value': stats['_all_systems_']['time_of_granted_99']},
    ]


@metric('grant_timings')
def compute_grant_timings_by_systems():
    return get_system_formatted_timings([ACTION.APPROVE], [ACTION.FAIL, ACTION.GRANT])


@metric('remove_timings')
def compute_remove_timings_by_systems():
    return get_system_formatted_timings([ACTION.DEPRIVE, ACTION.EXPIRE], [ACTION.REMOVE])


@metric('first_add_role_push_timings')
def compute_first_add_role_push_timings():
    return get_system_formatted_timings([ACTION.APPROVE], [ACTION.FIRST_ADD_ROLE_PUSH])


@metric('first_remove_role_push_timings')
def compute_first_remove_role_push_timings():
    return get_system_formatted_timings([ACTION.DEPRIVE, ACTION.EXPIRE], [ACTION.FIRST_REMOVE_ROLE_PUSH])


@metric('depriving_validation_timings')
def compute_depriving_validation_timings():
    validation_start_query = Q(data__force_deprive=False)
    validation_finish_query = Q(data__force_deprive=True)
    return get_system_formatted_timings(
        [ACTION.DEPRIVE],
        [ACTION.DEPRIVE],
        start_query=validation_start_query,
        finish_query=validation_finish_query,
        start_predictor=lambda action: not action.get('data', {}).get('force_deprive', False)
    )
