import logging
import yenv
import pytz

from yql.api.v1.client import YqlClient

from django.conf import settings
from django.utils import timezone

from idm.core.management.base import IdmBaseCommand
from idm.utils.i18n import set_translation
from idm.utils.lock import lock

log = logging.getLogger(__name__)


MINUTE_LOG = '5_min'
DAY_LOG = '1_day'


class MetricBaseCommand(IdmBaseCommand):
    """
    Базовый класс для подсчета и загрузки в метрику данных из логов
    """

    SYNTAX_VERSION = 1  # SQLv1. Да, надо переходить на новую версию

    STAT_CLUSTERS = {
        'testing': 'stat_beta',
        'production': 'stat',
    }

    LOG_LAG = {
        MINUTE_LOG: 30,  # 5-минутные логи отгружаем с задержкой в 30 минут (время в минутах)
        DAY_LOG: 24 * 60,  # Дневные логи отгружаем с задержкой в один день (время в минутах)
    }

    DATE_FORMAT = {
        MINUTE_LOG: '%Y-%m-%dT%H:%M:%S',
        DAY_LOG: '%Y-%m-%d',
    }

    MSK_TZ = pytz.timezone('Europe/Moscow')

    LOG_TYPES = (MINUTE_LOG, DAY_LOG,)
    DEFAULT_LOG_TYPE = DAY_LOG
    SKIP_IN_TESTING = False

    def add_arguments(self, parser):
        super(MetricBaseCommand, self).add_arguments(parser)
        parser.add_argument(
            '--log_type',
            action='store',
            dest='log_type',
            choices=self.LOG_TYPES,
            required=False,
            default=self.DEFAULT_LOG_TYPE
        )

    def get_table_name(self, log_type, report_date=None):
        if report_date is None:
            report_date = timezone.now() - timezone.timedelta(minutes=self.LOG_LAG[log_type])
        if log_type == MINUTE_LOG:
            truncated_minutes = report_date.minute - report_date.minute % 5
            report_date = report_date.replace(minute=truncated_minutes, second=0).astimezone(self.MSK_TZ)
        return report_date.strftime(self.DATE_FORMAT[log_type])

    def get_yql_query(self, log_type, table_name):
        raise NotImplementedError()

    def calculate_response_time(self, log_type, table_name):
        full_query = self.get_yql_query(log_type, table_name)
        yql_client = YqlClient(token=settings.IDM_ROBOT_YQL_OAUTH_TOKEN)
        request = yql_client.query(full_query, syntax_version=self.SYNTAX_VERSION)
        request.run()
        for table in request.get_results():
            table.fetch_full_data()
        log.info('Command %s for %s sent data to Statface', self.__class__.__module__, table_name)

    def idm_handle(self, *args, table_name, **options):
        try:
            self.calculate_response_time(options.get('log_type'), table_name)
        except Exception:
            log.exception('Command %s can not push data to Statface', self.__class__.__module__)

    @set_translation
    def handle(self, *args, **options):
        # Переопределяем метод базового класса для изменения имени лока
        # берем лок по имени таблицы, по которой собираем статистику
        if self.SKIP_IN_TESTING and yenv.type != 'production':
            log.info(f'Skipping command in test env for {self.lock_name}')
            return 
        table_name = self.get_table_name(log_type=options.get('log_type'))
        with lock(f'{self.lock_name}.{table_name}', block=options['use_block_lock']) as locked:
            if not locked:
                return
            self.stamped_run(*args, table_name=table_name, **options)
