# coding: utf-8
import copy
import logging
from collections import defaultdict
from datetime import datetime, timedelta

import pytz
import yenv
from constance import config
import statface_client
from django.conf import settings
from statface_client import StatfaceHttpResponseError
from yql.api.v1.client import YqlClient

from idm.core.management.base import IdmBaseCommand

logger = logging.getLogger(__name__)

yql_client = YqlClient(token=settings.IDM_ROBOT_YQL_OAUTH_TOKEN)


class StatfaceBaseCommand(IdmBaseCommand):
    stat_config = {
        'oauth_token': settings.IDM_ROBOT_YQL_OAUTH_TOKEN,
        'host': settings.STATFACE_HOST
    }

    moscow_tz = pytz.timezone('Europe/Moscow')
    client = statface_client.StatfaceClient(client_config=stat_config)
    report = client.get_report('ExtData/idm/api/v1')
    select_fields = []
    subselect_fields = [
        '$capture_url(request) as request',
        '$date_formatter(ts) as fielddate',
        'IF(CAST(status as int32) < 499, 0, 1) as bad_response_flag',
    ]
    group_by_fields = ['request', 'fielddate']

    api_v1_urls = [
        'approverequests',
        'batch',
        'firewall',
        'groups',
        'inconsistencies',
        'rolenodes',
        'rolerequests',
        'roles',
        'transfers',
        'users',
    ]

    query = """
-- idm_api_v1_error_distribution(error_methods, good_vs_bad)
USE HAHN;
$script=@@
from datetime import datetime
from pytz import timezone
import pytz

moscow = timezone('Europe/Moscow')

def date_formatter(ts):
    if ts:
        try:
            ts = int(float(ts))
            ts = ts - ts % 300
            return datetime.utcfromtimestamp(ts).replace(tzinfo=pytz.UTC).astimezone(moscow).strftime('%Y-%m-%d %H:%M:%S')
        except ValueError:
            return ts.split('T')[0]
    else:
        return ts
@@;

$date_formatter = Python::date_formatter(
    Callable<(String?)->String?>,
    $script
);

$capture_url = Pire::Capture("/api/v1/(.+)/.*");

SELECT {full_select}
FROM
(
    SELECT {full_subselect}

    FROM `home/logfeller/logs/qloud-router-log/stream/5min/{table_name}`

    WHERE
        qloud_project = "tools-access" AND
        qloud_application = "idm" AND
        qloud_environment = "{qloud_environment}"

)

WHERE request != "null"

GROUP BY
    {full_group_by};

    """

    @staticmethod
    def get_current_time(report_date=None):
        """
            Возвращает текущее время - DELAY_MINUTES в нужном формате для statface
        """
        if report_date is None:
            report_date = datetime.now(StatfaceBaseCommand.moscow_tz)
        current_minute = report_date.minute
        current_minute -= current_minute % 5
        return (
            report_date.
            replace(minute=current_minute, second=0, microsecond=0) - timedelta(minutes=config.IDM_API_V1_YT_LOGS_DELAY)
        ).strftime('%Y-%m-%dT%H:%M:%S')

    def get_initial_values(self):
        raise NotImplementedError()

    def get_data_from_yt(self):
        current_time = self.get_current_time()
        full_select = ','.join(self.select_fields)
        full_group_by = ','.join(self.group_by_fields)
        full_subselect = ','.join(self.subselect_fields)
        full_query = self.query.format(
            full_select=full_select,
            full_subselect=full_subselect,
            table_name=current_time,
            full_group_by=full_group_by,
            qloud_environment=yenv.type,
        )

        request = yql_client.query(full_query, syntax_version=1)
        request.run()
        return request

    def initialize_dict(self):
        initial_values = self.get_initial_values()
        data = defaultdict(lambda: copy.deepcopy(initial_values))

        current_minute = datetime.now(self.moscow_tz).minute
        current_minute -= current_minute % 5
        fielddate = (datetime.now(self.moscow_tz).replace(minute=current_minute, second=0, microsecond=0) -
                     timedelta(minutes=config.IDM_API_V1_YT_LOGS_DELAY)).strftime('%Y-%m-%d %H:%M:%S')

        data[fielddate] = copy.deepcopy(initial_values)
        return data

    def get_final_result(self, data):
        raise NotImplementedError()

    def populate_data(self, data, request):
        if request.errors:
            logger.info('[%s] Error fetching data from yt: %s', self.TASK_NAME, str(request))
            return
        else:
            for table in request.get_results():
                table.fetch_full_data()
                for row in table.rows:
                    raw_dict = {}
                    for name, value in zip(table.column_names, row):
                        raw_dict[name] = value
                    yield raw_dict

    def upload_to_statface(self, result):
        try:
            logger.info('{} started'.format(self.TASK_NAME))
            self.report.upload_data(scale='i', data=result, _append_mode=1)
            logger.info('{} completed'.format(self.TASK_NAME))
        except StatfaceHttpResponseError as exc:
            logger.exception('{}: {}'.format(self.TASK_NAME, str(exc)))

    def idm_handle(self, *args, **options):

        request = self.get_data_from_yt()
        data = self.initialize_dict()
        self.populate_data(data, request)
        if not request.errors:
            result = self.get_final_result(data)
            self.upload_to_statface(result)
