import yenv

from idm.metrics.management.commands.base import StatfaceBaseCommand, yql_client


class Command(StatfaceBaseCommand):

    TASK_NAME = 'BatchGoodAndBadRequestCounter'
    report = StatfaceBaseCommand.client.get_report('ExtData/idm/api/batch_good_vs_bad')
    query = """
-- idm_api_v1_batch_requests_count
USE HAHN;
PRAGMA yson.DisableStrict;

$script=@@
from datetime import datetime
from pytz import timezone
import pytz

moscow = timezone('Europe/Moscow')

def date_formatter(ts):
    try:
        current_datetime = datetime.strptime(ts, "%Y-%m-%d %H:%M:%S")
        truncated_minutes = current_datetime.minute - current_datetime.minute % 5
        return current_datetime.replace(minute=truncated_minutes, second=0).strftime('%Y-%m-%d %H:%M:%S')
    except ValueError:
        return ts.split(' ')[1]
@@;

$date_formatter = Python::date_formatter(
    Callable<(String?)->String?>,
    $script
);

$request_path = ($fields) -> {{ return Yson::ConvertToString($fields.context.request.path); }};
$final_flag = ($fields) -> {{ return Yson::ConvertToInt64($fields.context.profiling.final); }};
$status_code = ($fields) -> {{ return Yson::ConvertToInt64($fields.context.request.status_code); }};
$user_email = ($fields) -> {{ return Yson::ConvertToString($fields.context.user.email); }};

SELECT fielddate,
       sum(case when bad_response_flag = 0 then 1 else 0 end) as good_responses,
       sum(case when bad_response_flag = 1 then 1 else 0 end) as bad_responses
FROM
(
    SELECT 
        $request_path(fields) as request_path,
        $date_formatter(iso_eventtime) as fielddate,
        IF($status_code(fields) < 499, 0, 1) as bad_response_flag

    FROM `home/logfeller/logs/qloud-runtime-log/stream/5min/{table_name}`

    WHERE
        qloud_project = "tools-access" AND
        qloud_application = "idm" AND
        qloud_environment = "{qloud_environment}" AND
        $user_email(fields) = 'cauth-robot-rw@yandex-team.ru' AND
        $request_path(fields) = '/api/v1/batch/' AND
        $final_flag(fields) = 1
)


GROUP BY
    fielddate;

        """

    def get_initial_values(self):
        return (0, 0)

    def get_data_from_yt(self):
        current_time = self.get_current_time()
        full_query = self.query.format(
            table_name=current_time,
            qloud_environment=yenv.type
        )

        request = yql_client.query(full_query, syntax_version=1)
        request.run()
        return request

    def populate_data(self, data, request):
        for raw_dict in super(Command, self).populate_data(data, request):
            data[raw_dict['fielddate']] = (raw_dict['good_responses'], raw_dict['bad_responses'])

    def get_final_result(self, data):
        result = []
        for fielddate, (good_responses, bad_responses) in data.items():
            final_row = {
                'fielddate': fielddate,
                'good_responses': good_responses,
                'bad_responses': bad_responses
            }
            result.append(final_row)

        return result
