import itertools
from collections import defaultdict
from typing import List, Iterable

from metrics_framework.decorators import metric
from django.utils import timezone

from idm.core.models import ActionTiny
from idm.core.constants.action_tiny import ACTION_TINY, STATUS


def get_metric_values(seq: Iterable, prefix: str) -> List[dict]:
    result = []
    sorted_seq = sorted(seq)
    for p in (50, 75, 90, 99):
        result.append({
            'slug': f'{prefix}_{p}',
            'value': sorted_seq[int(len(sorted_seq) * p / 100 - 1)] if sorted_seq else 0.0
        })

    result.append({
        'slug': 'count',
        'value': len(sorted_seq),
    })

    return result


@metric('useful_pipelines_time')
def useful_pipelines_time():
    actions = ActionTiny.objects.filter(
        action=ACTION_TINY.PIPELINE_FINISHED,
        finish__gte=timezone.datetime.now() - timezone.timedelta(days=1),
        status=STATUS.OK,
    )

    system_to_times = defaultdict(list)
    for system_slug, start, finish in actions.values_list('system__slug', 'start', 'finish').iterator():
        system_to_times[system_slug].append((finish - start).total_seconds())

    result = []
    for system_slug, times in system_to_times.items():
        result.append({
            'context': {
                'system': system_slug,
            },
            'values': get_metric_values(times, 'time'),
        })

    result.append({
        'context': {
            'system': '__ALL__',
        },
        'values': get_metric_values(itertools.chain(*system_to_times.values()), 'time'),
    })

    return result
