from typing import Optional

from django.utils import timezone
from metrics_framework.decorators import metric
from metrics_framework.models import MetricPoint

from idm.core.models import System
from idm.services.managers import ServiceQueryset
from idm.services.models import Service
from idm.users.constants.group import GROUP_TYPES
from idm.users.models import Group, GroupMembership


def get_active_descendants(root: Optional[Service]) -> ServiceQueryset:
    active_services = Service.objects.active_in_abc()
    if root is not None:
        active_services = active_services.filter(serviceclosure_parents__parent=root)

    return active_services


def get_metric_key(root: Optional[Service]) -> str:
    return root.slug if root is not None else '__ALL__'


def get_descendants_with_tvm_count(root: Optional[Service]) -> int:
    descendants_slugs = get_active_descendants(root).values_list('slug', flat=True)
    groups = Group.objects.filter(type=GROUP_TYPES.TVM_SERVICE, slug__in=descendants_slugs)
    services_with_tvm = GroupMembership.objects.active().filter(group__in=groups).values('group_id').distinct()
    return services_with_tvm.count()


def get_systems_with_abc_count(root: Optional[Service]) -> int:
    descendants = get_active_descendants(root)
    return System.objects.active().filter(service__in=descendants).count()


def get_diff_values(root: Service, current_value: dict) -> dict:
    old_metric_point = (
        MetricPoint.objects
        .filter(
            metric__slug='tvm_stat',
            context__root=get_metric_key(root),
            created_at__date=(timezone.now() - timezone.timedelta(weeks=1)).date(),
        )
        .first()
    )
    if old_metric_point is not None:
        result = {}
        old_values = {v.slug: v.value for v in old_metric_point.values.all()}
        for k, v in list(current_value.items()):
            result[k + '_diff'] = current_value[k] - old_values[k] if k in old_values else None
    else:
        result = {k + '_diff': None for k in current_value}

    return result


@metric('tvm_stat')
def tvm_stat():
    result = []
    roots = Service.objects.active_in_abc().actual_roots()
    for root in (None, *roots):
        metric_value = {
            'total_abc_with_tvm': get_descendants_with_tvm_count(root),
            'total_active_systems_with_abc': get_systems_with_abc_count(root),
        }
        metric_value.update(get_diff_values(root, metric_value))

        result.append({
            'context': {
                'root': get_metric_key(root),
            },
            'values': [{'slug': k, 'value': v} for k, v in metric_value.items()],
        })

    return result
