import os
import socket
from enum import Enum, EnumMeta
from typing import List, Iterable, Any, Dict

import requests
import yenv
from attr import dataclass, attrib
from django.conf import settings


def get_current_host() -> str:
    return os.getenv('PORTO_HOST', socket.gethostname())


def get_current_instance() -> str:
    return os.getenv('DEPLOY_UNIT_ID', '')


DEFAULT_TAGS = ('backend', yenv.name)


class JugglerStatus(str, Enum):
    OK = 'OK'
    CRITICAL = 'CRIT'
    WARNING = 'WARN'

    def __iter__(self) -> Iterable['JugglerStatus']:
        return EnumMeta.__iter__(self.__class__)

    def __str__(self):
        return self.value

    __repr__ = __str__


@dataclass(slots=True)
class JugglerEvent:
    service: str
    status: JugglerStatus
    description: str = ""
    host: str = attrib(factory=get_current_host)
    instance: str = attrib(factory=get_current_instance)
    tags: List[str] = attrib(factory=list)

    def __attrs_post_init__(self):
        self.tags = list({*self.tags, *DEFAULT_TAGS})

    def as_dict(self) -> Dict[str, Any]:
        return {attr.name: getattr(self, attr.name) for attr in self.__attrs_attrs__}


class JugglerClient:
    def __init__(
            self,
            url: str = settings.JUGGLER_URL,
            app_name: str = settings.JUGGLER_APPLICATION_NAME,
            oauth_token: str = settings.JUGGLER_OAUTH_TOKEN,
    ):
        assert url and app_name
        self.url = url
        self.app_name = app_name
        self._oauth_token = oauth_token

    def push_events(self, *events: JugglerEvent):
        headers = {}
        if self._oauth_token:
            headers['Authorization'] = f'OAuth {self._oauth_token}'
        response = requests.post(self.url, headers=headers, json={
            "source": self.app_name,
            "events": [
                event.as_dict()
                for event in events
            ],
        })
        response.raise_for_status()
