import collections

import constance
from django.db.models import Count, Q
from django.utils import timezone
from django.utils.functional import cached_property

from idm.celery_app import app
from idm.core.constants.groupmembership_system_relation import MEMBERSHIP_SYSTEM_RELATION_STATE
from idm.core.models import System, Role, GroupMembershipSystemRelation, UserPassportLogin
from idm.framework.task import BaseTask
from idm.monitorings.juggler import JugglerClient, checks
from idm.monitorings.metric import RoleStatesMetric, GroupMembershipStatesMetric, InconsistencyStatesMetric, \
    PassportLoginsToSubscribeMetric, LastSyncStatesMetric
from idm.utils.lock import lock


class CalculateUnistatMetrics(BaseTask):
    max_retries = 0

    def init(self):
        with lock(f'idm.monitorings.tasks.CalculateUnistatMetrics') as acquired:
            if not acquired:
                return
            RoleStatesMetric.set(self.get_role_states())
            GroupMembershipStatesMetric.set(self.get_groupmembership_system_states())
            InconsistencyStatesMetric.set(self.get_inconsistency_states())
            PassportLoginsToSubscribeMetric.set(self.get_logins_to_subscribe())
            LastSyncStatesMetric.set(self.get_last_sync_states())

    @cached_property
    def system_id_to_slug(self):
        return dict(System.objects.values_list('id', 'slug'))

    def get_logins_to_subscribe(self):
        return UserPassportLogin.objects.to_subscribe().count()
    
    def get_last_sync_states(self):
        """
        Отдает количество часов с момента окончания последнего успешного завершения 
        синхронизации ролей (everysync all_steps) по системам
        """
        metrics = []
        now = timezone.now()
        systems = System.objects.select_related('metainfo').filter(
            slug__in=constance.config.SYSTEM_SLUGS_FOR_LAST_SYNC_UNISTAT.split(','),
            is_active=True,
        )
        
        for system in systems:
            hours_passed = (now - system.metainfo.last_check_inconsistencies_finish).total_seconds() / 3600
            metrics.append([f'last_sync_in_{system.slug}_max', round(hours_passed, 2)])

        return metrics

    def get_inconsistency_states(self):
        """
        Посчитаем количество расхождений в каждой включенной системе
        """
        metrics = []
        
        system_inconsistencies = System.objects.filter(
            slug__in=constance.config.SYSTEM_SLUGS_FOR_INCONSISTENCY_UNISTAT.split(','),
            is_active=True,
        ).values('slug').annotate(
            inc_count=Count(
                'inconsistencies', 
                filter=Q(inconsistencies__state='active'),
            )
        )

        for system_data in system_inconsistencies:
            metrics.append([f'inconsistencies_in_{system_data["slug"]}_max', system_data['inc_count']])
        
        return metrics
    
    def get_role_states(self):
        metrics = []

        # unistat не умеет обрабатывать такое количество сигналов, пока по каждой системе не отправляем
        # TODO: уточнить про это ограничение и, возможно, перейти на push, если правда есть необходимость
        # в графиках по каждой системе
        # system_states = Role.objects.values('system', 'state').annotate(cnt=Count('*')).order_by()
        # for system_state in system_states:
        #     system_slug = self.system_id_to_slug[system_state['system']]
        #     state = system_state['state']
        #     metrics.append([f'ctype={system_slug};role_{state}_max', system_state['cnt']])

        role_states = Role.objects.values('state').annotate(cnt=Count('*')).order_by()
        for role_state in role_states:
            state = role_state['state']
            metrics.append([f'role_{state}_max', role_state['cnt']])

        return metrics

    def get_groupmembership_system_states(self):
        metrics = []

        state_cnt = dict.fromkeys(MEMBERSHIP_SYSTEM_RELATION_STATE.STATUSES, 0)
        system_state_cnt = collections.defaultdict(state_cnt.copy)

        system_states = GroupMembershipSystemRelation.objects.values('system', 'state').annotate(cnt=Count('*')).order_by()
        for system_state in system_states:
            system_slug = self.system_id_to_slug[system_state['system']]
            system_state_cnt[system_slug][system_state['state']] = system_state['cnt']

        for system_slug, states in system_state_cnt.items():
            for state, count in states.items():
                metrics.append([f'ctype={system_slug};groupmembership_system_{state}_max', count])

        return metrics


class PushJugglerEvents(BaseTask):
    max_retries = 0

    def init(self):
        if not checks.ACTIVE_CHECKS:
            raise RuntimeError('Not any active juggler check')
        with lock(f'idm.monitorings.tasks.PushJugglerEvents') as acquired:
            if not acquired:
                self.log.debug('Couln\'t acquire lock, seems like is already acquired.')
                return
            events = []
            for check in checks.ACTIVE_CHECKS:
                event = check.get_event()
                events.append(event)
                self.log.debug(f'Get status of check {check.name}: {event.status}')

            client = JugglerClient()
            client.push_events(*events)


CalculateUnistatMetrics: BaseTask = app.register_task(CalculateUnistatMetrics())
PushJugglerEvents: BaseTask = app.register_task(PushJugglerEvents())
