# coding: utf-8


import abc
from collections import defaultdict

from django.conf import settings
from django.db.models import Count
from django.utils import timezone
import constance

from idm.monitorings.views.base import BaseMonitoringView
from idm.monitorings.answer import WarnAnswer, CritAnswer
from idm.core.models import Role


class BaseHangingRolesView(BaseMonitoringView):
    TOP_SYSTEMS_COUNT = 5

    @abc.abstractmethod
    def get_queryset(self):
        pass

    @property
    @abc.abstractmethod
    def role_type(self):
        pass

    def error_messages(self, count, since, top_systems):
        return ['IDM has {} {} hanging roles since {}'.format(count, self.role_type, since)] + [
            'System {system__slug} has {count}'.format(**x) for x in top_systems]

    def monitoring(self):
        hanging_roles = self.get_queryset()

        if hanging_roles:
            oldest = min([role.last_action_added for role in hanging_roles])
            top_systems = hanging_roles.values('system__slug').annotate(
                count=Count('id', distinct=True)
            ).order_by('-count')[:self.TOP_SYSTEMS_COUNT]
            hanging_time = (timezone.now() - oldest).total_seconds() / 60
            if hanging_time < constance.config.HANGING_ROLES_WARN_MINUTES:
                return WarnAnswer(self.error_messages(hanging_roles.count(), oldest.strftime('%F %T'), top_systems))
            return CritAnswer(self.error_messages(hanging_roles.count(), oldest.strftime('%F %T'), top_systems))

        return []


class HangingApprovedRolesView(BaseHangingRolesView):
    def get_queryset(self):
        systems = self.request.GET.get('systems')
        threshold = self.request.GET.get('threshold')

        result = Role.objects.hanging_approved(threshold=threshold)

        if systems is not None:
            result = result.filter(system__slug__in=systems.split(','))

        return result

    @property
    def role_type(self):
        return 'approved'


class HangingDeprivingRolesView(BaseHangingRolesView):
    def get_queryset(self):
        queryset = Role.objects.hanging_depriving()
        if self.check_if_applicable(queryset):
            return queryset
    
    def check_if_applicable(self, queryset):
        by_system = defaultdict(int)
        for item in queryset:
            by_system[item.system_id] += 1
        
        if by_system and (    
            len(by_system) > settings.IDM_HANGING_DEPRIVING_SYSTEMS_THRESHOLD 
            or max(by_system.values()) > settings.IDM_HANGING_DEPRIVING_BY_SYSTEM_THRESHOLD
        ):
            return True

    @property
    def role_type(self):
        return 'depriving'
