# coding: utf-8
from __future__ import absolute_import, unicode_literals

import collections
import json
from itertools import chain
from typing import Tuple, List

from django.conf import settings
from django.http.response import HttpResponse
from django.utils import timezone
from django.views.generic import View

from idm.core.models import CommandTimestamp
from idm.monitorings.metric import QueueSizeMetric, PeriodicTimestampsMetric, RoleStatesMetric, \
    InconsistencyStatesMetric, LastSyncStatesMetric, FailedDeprivingRolesMetric, TotalDeprivingRolesMetric, \
    GroupMembershipStatesMetric
from idm.monitorings.tasks import CalculateUnistatMetrics
from idm.utils.mongo import get_mongo_db, get_queue_size


class UnistatView(View):
    @staticmethod
    def get_queues_size() -> List[Tuple[str, float]]:
        if metrics := QueueSizeMetric.get():
            return metrics

        mongo_db = get_mongo_db()
        queue_size_metric_suffix = '_queue_size_max'
        queue_names = [queue.name for queue in settings.CELERY_QUEUES]

        metrics = [
            (f'{name}{queue_size_metric_suffix}', get_queue_size(name, mongo_db)) for name in queue_names
        ]
        QueueSizeMetric.set(metrics)
        return metrics

    @staticmethod
    def get_periodic_timestamps() -> List[Tuple[str, float]]:
        if metrics:= PeriodicTimestampsMetric.get():
            return metrics

        SUFFIX = '_freshness_max'

        by_task_or_command = collections.defaultdict(list)
        for timestamp in CommandTimestamp.objects.all():
            parts = timestamp.command.split(':')
            task_type = parts[0]
            by_task_or_command[task_type].append(timestamp.last_success_finish)

        metrics = []
        for task_type in by_task_or_command:
            min_timestamp = min(by_task_or_command[task_type])
            task_freshness = round((timezone.now() - min_timestamp).total_seconds() / 3600, 2)  # в часах
            metric_name = task_type + SUFFIX
            metrics.append([metric_name, task_freshness])

        PeriodicTimestampsMetric.set(metrics)
        return metrics

    @staticmethod
    def get_role_states() -> List[Tuple[str, float]]:
        return RoleStatesMetric.get() or []
    
    @staticmethod
    def get_inconsistency_states() -> List[Tuple[str, float]]:
        return InconsistencyStatesMetric.get() or []
    
    @staticmethod
    def get_last_sync_states() -> List[Tuple[str, float]]:
        return LastSyncStatesMetric.get() or []

    @staticmethod
    def get_groupmembership_system_states() -> List[Tuple[str, float]]:
        return GroupMembershipStatesMetric.get() or []

    @staticmethod
    def get_depriving_roles_states() -> List[Tuple[str, float]]:
        return [
            ('failed_depriving_roles_max', FailedDeprivingRolesMetric.get() or 0),
            ('total_depriving_roles_max', TotalDeprivingRolesMetric.get() or 0),
        ]

    def get(self, request):
        all_metrics = list(chain(
            self.get_queues_size(),
            self.get_periodic_timestamps(),
            self.get_role_states(),
            self.get_groupmembership_system_states(),
            self.get_inconsistency_states(),
            self.get_last_sync_states(),
            self.get_depriving_roles_states(),
        ))

        return HttpResponse(
            json.dumps(all_metrics),
            content_type='text/json',
            status=HttpResponse.status_code,
        )
