import constance

from django.db.models import Q
from django.utils import timezone

from idm.core.models import System, SystemMetainfo
from idm.monitorings.views.base import BaseMonitoringView


def _get_sync_nodes_queryset():
    return (
        System
        .objects
        .get_operational()
        .filter(
            nodes__is_auto_updated=True,
            nodes__state='active',
        )
    )


def _get_default_queryset():
    return (
        System
        .objects
        .get_operational()
    )

def _get_memberships_aware_queryset():
    return (
        System
        .objects
        .get_operational()
        .get_systems_with_group_sync_policy()
    )


def unfold(d):
    """Метод, который берёт словарь вида {(a, b, c) -> v} и разворачивает его в {a->v, b->v, c->v]"""
    result_dict = {}
    for keys_list, value in d.items():
        if isinstance(keys_list, str):
            result_dict[keys_list] = value
            continue
        for key in keys_list:
            result_dict[key] = value
    return result_dict


class UnsynchronizedSystemsView(BaseMonitoringView):
    TASK_TO_QUERYSET = unfold({
        'sync_nodes': _get_sync_nodes_queryset,
        'default': _get_default_queryset,
        ('activate_memberships', 'deprive_memberships', 'update_memberships',
         'check_memberships', 'resolve_memberships'): _get_memberships_aware_queryset,
    })
    TASK_TO_THRESHOLD = unfold({
        (
            'activate_memberships',
            'deprive_memberships',
            'update_memberships',
            'recalc_pipeline',
        ): int(constance.config.UNSYNCHRONIZED_REGULAR_SYSTEMS_DELTA_MINUTES),
        'default': int(constance.config.UNSYNCHRONIZED_SYSTEMS_DELTA_DAYS) * 24 * 60,  # переводим в минуты
    })

    def monitoring(self):
        start_time = timezone.now()
        errors = []
        for task_type in sorted(SystemMetainfo.TRACKABLE_TASKS):
            queryset_func = self.TASK_TO_QUERYSET.get(task_type) or self.TASK_TO_QUERYSET['default']
            threshold = self.TASK_TO_THRESHOLD.get(task_type) or self.TASK_TO_THRESHOLD['default']
            since_threshold = start_time - timezone.timedelta(minutes=threshold)
            min_system_age = start_time - timezone.timedelta(days=int(constance.config.NEW_SYSTEMS_MONITORINGS_DOWNTIME))

            timestamp_field = 'metainfo__last_{}_finish__lt'.format(task_type)
            timestamp_isnull = 'metainfo__last_{}_finish__isnull'.format(task_type)
            should_monitor_field = 'metainfo__monitor_{}'.format(task_type)

            failed_systems = (
                queryset_func()
                .filter(**{should_monitor_field: True})
                .filter(
                    Q(**{timestamp_field: since_threshold}) |
                    (Q(**{timestamp_isnull: True}) & Q(added__lte=min_system_age))
                )
                .values_list('slug', flat=True)
                .distinct()
            )

            if len(failed_systems) > 0:
                errors += ['"{}" task has not been recently run for systems {}'.format(
                    task_type,
                    ','.join(sorted(failed_systems))
                )]

        return errors
