# coding: utf-8

import collections
import logging

from django.db import models, transaction
from django.utils.translation import ugettext_lazy as _

from idm.closuretree.models import ClosureModel
from idm.nodes.hashers import Hasher
from idm.nodes.queue import Queue
from idm.utils.log import log_duration
from idm.utils.model_helpers import raise_if_unsaved


log = logging.getLogger(__name__)


class StatefulQuerySet(models.QuerySet):
    def active(self):
        return self.filter(state__in=self.model.ACTIVE_STATES)

    def inactive(self):
        return self.filter(state__in=self.model.INACTIVE_STATES)

    def depriving(self):
        return self.filter(state='depriving')


class UpdatableNode(ClosureModel):
    """Модель узла, который может быть синхронизирован с удалённым источником."""
    ACTIVE = 'active'
    DEPRIVING = 'depriving'
    DEPRIVED = 'deprived'

    STATES = (
        (ACTIVE, _('Активен/активна')),
        (DEPRIVING, _('Удаляется')),
        (DEPRIVED, _('Удалён/удалена'))
    )
    ACTIVE_STATES = (ACTIVE,)
    RETURNING_STATES = (ACTIVE, DEPRIVING)
    INACTIVE_STATES = (DEPRIVING, DEPRIVED)

    SEP = '/'

    state = models.CharField(_('Состояние'), choices=STATES, max_length=255, default='active', db_index=True)
    path = models.CharField(max_length=255, db_index=True, verbose_name=_('Путь'), blank=True)
    hash = models.CharField(max_length=255, verbose_name=_('Хеш от внутренних данных и хешей дочерних узлов'),
                            blank=True, db_index=True, default='')
    slug = models.CharField(_('Слаг'), max_length=255, default='', db_index=True)
    parent = models.ForeignKey('self', null=True, blank=True, related_name='children', on_delete=models.CASCADE)
    descendants_count = models.PositiveIntegerField(verbose_name=_('Количество потомков'), null=True, blank=True,
                                                    default=None)

    hasher = Hasher(debug=False)
    queue_class = Queue

    class Meta:
        abstract = True

    def is_active(self):
        return self.state in self.ACTIVE_STATES

    def mark_depriving(self, **kwargs):
        descendants = self.get_descendants(include_self=True, **kwargs)
        descendants.update(state='depriving', hash='')
        self.state = 'depriving'
        self.hash = ''
        return self

    def get_child_nodes(self, include_depriving=False):
        children = self.get_children()
        if include_depriving:
            children = children.filter(state__in=('active', 'depriving'))
        else:
            children = children.filter(state='active')
        return children

    @raise_if_unsaved
    def get_children(self):
        """Версия без кеша и полагающаяся только на parent"""
        return self.__class__._default_manager.filter(parent=self)

    def find_matching_child(self, data_item, nodes):
        raise NotImplementedError()

    def build_path(self, for_parent=Ellipsis):  # используем Ellipsis, чтобы отличить None от NotProvided
        if for_parent is not Ellipsis:
            parent = for_parent
        else:
            parent = self.parent
        if parent is None:
            path = '%s%s%s' % (self.SEP, self.slug, self.SEP)
        else:
            path = '%s%s%s' % (parent.path, self.slug, self.SEP)
        return path

    def rehash(self):
        assert self.level == 0
        # сбросим хеш у всех предков узлов, у которых хеш уже сброшен
        emptyhash_descendants = self.get_descendants(include_self=True).active().filter(hash='')
        ancestors_of_emptyhash = self._toplevel().objects.active().filter(**{
            '%s__child__in' % self._closure_parentref(): emptyhash_descendants
        })
        ancestors_of_emptyhash.update(hash='')
        children_of_emptyhash = (
            self.get_descendants().filter(parent__hash='').exclude(hash='').values_list('parent_id', 'hash')
        )
        hashdict = collections.defaultdict(list)
        for parent_id, hash_ in children_of_emptyhash:
            hashdict[parent_id].append(hash_)

        for node in emptyhash_descendants.prefetch_for_hashing().order_by('-level'):
            hash_ = self.hasher.hash_object(node, children_hashes=hashdict.get(node.pk, ()))
            hashdict[node.parent_id].append(hash_)
            node.hash = hash_
            node.save(update_fields=('hash',))

    def get_prefetched_descendants(self):
        return self.get_descendants(include_self=True).active().order_by('level', 'pk')

    def save(self, *args, **kwargs):
        recalc_fields = kwargs.pop('recalc_fields', False)
        if not self.pk or recalc_fields:
            self.path = self.build_path()
        result = super(UpdatableNode, self).save(*args, **kwargs)
        return result

    def synchronize(self, **kwargs):
        model_name = self._meta.model_name
        with log_duration(log, 'Rehash nodes from <id %s, type %s>', self.pk, model_name):
            self.rehash()

        with log_duration(log, 'Get external nodes for <id %s, type %s>', self.pk, model_name):
            data = self.fetcher.fetch(self)
        if data is NotImplemented:
            return True, None

        with log_duration(log, 'Calculate changes for <id %s, type %s>', self.pk, model_name):
            queue = self.get_queue(data, **kwargs)
        if not queue:
            log.info('There were no changes for <id %s, type %s>', self.pk, model_name)
            return False, None

        with log_duration(log, 'Synchronize nodes for <id %s, type %s>', self.pk, model_name):
            queue.apply()
        return True, queue

    def get_queue_instance(self, **kwargs):
        queue = self.queue_class()
        return queue

    def get_queue(self, data, **kwargs):
        canonical = self.hasher.hash_canonical(data)
        queue = self.compare(canonical, **kwargs)
        return queue

    def compare(self, data, **kwargs):
        is_modified = False
        supports_restore = getattr(self, 'SUPPORTS_RESTORE', False)
        queue = self.get_queue_instance(**kwargs)
        if self.hash == data.hash:
            return queue
        canonical = self.as_canonical()
        if canonical != data:
            queue.push_modification(node=self, new_data=data, **kwargs)
            is_modified = True
        # здесь нужно сделать prefetch_related() всем вложенным структурам, например, полям
        tree_children = list(self.get_child_nodes(include_depriving=supports_restore).prefetch_for_hashing())
        used = set()
        for child_data in sorted(getattr(data, 'children', []), key=lambda child: 0 if hasattr(child, 'unique_id') and child.unique_id else 1):
            match = self.find_matching_child(child_data, tree_children)
            if match and match not in used:  # нашёлся соответствующий узел
                used.add(match)
                if match.state == 'depriving':
                    restoring_info = {'data': child_data}
                    queue.push_restore(node=match, extra=restoring_info, **kwargs)
                    is_modified = True
                subqueue = match.compare(child_data, **kwargs)
                if subqueue.items:
                    is_modified = True
                queue.extend(subqueue)
            else:  # узла не нашлось, добавим в очередь на создание
                queue.push_addition(node=self, child_data=child_data, **kwargs)
                is_modified = True
        if used != set(tree_children):
            # какие-то узлы есть в нашем дереве, но их нет среди нового дерева.
            # это значит, что нам нужно удалить наши узлы
            for child in set(tree_children) - used:
                queue.push_removal(node=child, **kwargs)
                is_modified = True
        if not (is_modified or self.state == 'depriving'):
            log.warning(
                '{} with pk={} was not modified, but was unexpectedly processed in queue generation'.format(
                    self.__class__.__name__,
                    self.pk,
                )
            )
        queue.push_hash_update(node=self, hash=data.hash)
        return queue

    def apply_queue(self, queue, **kwargs):
        # применяем очередь в следующем порядке: restore, add, modify, remove
        for queue_item in sorted(queue, key=lambda item: item.order):
            with transaction.atomic():
                queue_item.apply(**kwargs)
        return queue
