# -*- coding: utf-8 -*-

import logging
import yenv

from django.conf import settings
from django.core.mail.backends.smtp import EmailBackend as SmtpBackend
from django.utils.encoding import force_text

log = logging.getLogger(__name__)


def _process_email_in_testenv(message):
    """
    Чтобы не спамить людей с тестинга, пишем только на адреса из списка исключений.
    Автоматически в список исключений попадают email'ы, указанные в данных подключенных систем.
    """
    if yenv.type == 'testing':
        from idm.testenv.models import AllowedEmail
        allowed_emails = AllowedEmail.get_test_env_allowed_emails()
    else:
        allowed_emails = []

    if not set(message.to).issubset(allowed_emails):
        message.to = []
    if not set(message.cc).issubset(allowed_emails):
        message.body = '[Initial cc was: %s]\n' % message.cc + message.body
        message.cc = list(set(message.cc) & set(allowed_emails))


class Backend(SmtpBackend):
    """Этот бэкенд считает количество отправленных писем, чтобы
    можно было нарисовать их на графике мониторинга.

    Так же, он ставит в копию указанный в настройках email. Это нужно,
    чтобы отслеживать сбои в рассылке почты или оформлении писем после
    рефакторинга.
    """
    def send_messages(self, email_messages):
        # на всякий случай превращаем email_messages в список,
        # чтобы случайно не "съесть" итератор

        email_messages = list(email_messages)

        # считатаем для статистики количество писем
        for message in email_messages:
            message.subject = settings.EMAIL_SUBJECT_PREFIX + message.subject
            message.extra_headers['X-IDM-Environment'] = yenv.type

            if yenv.type != 'production':
                # если мы не на продакшне, все адресаты, не указанные в исключениях
                # заменяются на ящики команды
                _process_email_in_testenv(message)

            log.info(
                'Sending email with subject {subject} to {emails}'.format(
                    subject=force_text(message.subject),
                    emails=', '.join(map(force_text, message.to))
                )
            )

        # и отправляем как обычно
        return super(Backend, self).send_messages(email_messages)
