import constance
import logging

from django.conf import settings
from django.db.models import Q
from django_idm_api.compat import get_user_model
from django_idm_api.exceptions import UserNotFound
from django_idm_api.hooks import AuthHooks, RoleNotFound, RoleStream

from idm.users.constants.user import USER_TYPES
from idm.core.models import System, InternalRole, InternalRoleUserObjectPermission
from idm.permissions.utils import add_perms_by_role, remove_perms_by_role, get_all_permissions


log = logging.getLogger(__name__)


def get_system_subtree(system):
    data = {
        'name': {
            'ru': system.name,
            'en': system.name_en,
        },
        'roles': {
            'slug': 'role',
            'name': {
                'en': 'Role',
                'ru': 'Роль',
            },
            'help': {
                'en': 'Role effective inside the system',
                'ru': 'Роль внутри системы',
            },
            'values': {}
        }
    }

    values = data['roles']['values']

    for role, internal_role_info in settings.IDM_SYSTEM_ROLES:
        values[role] = dict(internal_role_info, **{'set': role})
    return {system.slug: data}


class InternalCommonRoleStream(RoleStream):
    """Внутренние общие роли"""

    name = 'internal_common_roles'

    def get_queryset(self):
        """Результат должен быть отсортирован по pk."""
        qs = (
            InternalRole
            .objects
            .filter(user_object_permissions__isnull=False, node__isnull=True)
            .values_list('pk', flat=True)
            .order_by('pk')
            .distinct()
        )

        return qs

    def values_list(self, pks):
        """Первым значением каждого элемента должен быть pk."""
        return (
            InternalRole
            .objects
            .filter(pk__in=pks)
            .values_list(
                'pk',
                'user_object_permissions__user__username', 'user_object_permissions__user__type',
                'role',
            )
            .distinct(
                'user_object_permissions__user__username',
                'user_object_permissions__user__type',
                'role',
            )
        )

    def row_as_dict(self, row):
        pk, username, subject_type, role = row

        return {
            'login': username,
            'subject_type': subject_type,
            'path': '/group/common/role/%s/' % role,
        }


class InternalSystemsRoleStream(RoleStream):
    """Внутренние системные роли"""

    name = "internal_systems_roles"

    def get_queryset(self):
        """Результат должен быть отсортирован по pk."""
        # Кастомная реализация хуков django-idm-api, выбираем только pk
        query = (
            Q(content_object__node__isnull=False) & (
                Q(content_object__node__responsibilities__is_active=False) |
                Q(content_object__node__responsibilities__isnull=True)
            )
        )

        return (
            InternalRoleUserObjectPermission
            .objects
            .filter(query)
            .values_list('pk', flat=True)
            .order_by('pk')
            .distinct()
        )

    def values_list(self, pks):
        """Первым значением каждого элемента должен быть pk."""
        # pks - набор pk объектов, которые нужно выбрать, а не queryset
        return (
            InternalRoleUserObjectPermission
            .objects
            .filter(pk__in=pks)
            .values_list(
                'pk',
                'user__username',
                'user__type',
                'content_object__role',
                'content_object__node__system__slug',
                'content_object__node__value_path')
            .distinct(
                'user__username',
                'user__type',
                'content_object__role',
                'content_object__node__system__slug',
                'content_object__node__value_path'
            )
        )

    def row_as_dict(self, row):
        pk, username, subject_type, role, system_slug, value_path = row

        return {
            'login': username,
            'subject_type': subject_type,
            'path': '/group/system/system_on/%s/role/%s/' % (system_slug, role),
            'fields': {
                'scope': value_path,
            }
        }


class Hooks(AuthHooks):
    """Эти три хука используются для добавления дополнительных ролей, используемых в IDM.

    Хуки используются в django_idm_api для модификации списка ролей
    Django проекта (добавляют роли администратора каждой из систем),
    и обработки добавления и удаления роли.
    """

    GET_ROLES_PAGE_SIZE = constance.config.IDM_INTERNAL_ROLES_PAGE_SIZE
    GET_ROLES_STREAMS = [InternalCommonRoleStream, InternalSystemsRoleStream]

    def info(self):
        """Возвращает роли IDM: общие и для каждой конкретной системы"""
        data = super(Hooks, self).info()

        # вводим иерархичность дерева ролей
        data['roles']['slug'] = 'group'
        data['roles']['name'] = {
            'en': 'Type',
            'ru': 'Тип',
        }
        roles = data['roles']['values'] = {}

        # добавляем группу Общие роли
        roles['common'] = {
            'name': {
                'en': 'Common',
                'ru': 'Общие',
            },
            'help': {
                'en': 'IDM global roles',
                'ru': 'Глобальные для всего IDM роли',
            },
            'roles': {
                'slug': 'role',
                'name': {
                    'en': 'Role',
                    'ru': 'Роль',
                },
                'values': {}
            }
        }

        for role, global_role_info in settings.IDM_COMMON_ROLES:
            roles['common']['roles']['values'][role] = global_role_info

        # добавляем группу Система, в которой у каждой системы несколько ролей
        systems = System.objects.all()
        if systems.exists():
            roles['system'] = {
                'name': {
                    'en': 'Systems',
                    'ru': 'Системы',
                },
                'roles': {
                    'slug': 'system_on',
                    'name': {
                        'en': 'System',
                        'ru': 'Система',
                    },
                    'fields': [{
                        'slug': 'scope',
                        'type': 'charfield',
                        'name': {
                            'en': 'Scope',
                            'ru': 'Область',
                        },
                        'required': True,
                        'options': {
                            'default': '/',
                        }
                    }],
                    'values': {},
                }
            }

        for system in systems:
            system_subtree = get_system_subtree(system)
            roles['system']['roles']['values'].update(system_subtree)
        return data

    def get_all_roles(self):
        return {'code': 1, 'error': 'get-all-roles is not implemented'}

    def _get_user(self, login, subject_type=USER_TYPES.USER):
        """
        Получить пользователя по логину.
        Повторяет метод из django_idm_api, но умеет разделять обычных юзеров и tvm-приложения
        """
        try:
            return get_user_model().objects.get(username=login, type=subject_type)
        except get_user_model().DoesNotExist:
            raise UserNotFound('User does not exist: %s' % login)

    def add_role_impl(self, login, role, fields, **kwargs):
        return self._add_role_common_impl(login, USER_TYPES.USER, role, fields)

    def add_tvm_role_impl(self, login, role, fields, **kwargs):
        return self._add_role_common_impl(login, USER_TYPES.TVM_APP, role, fields)

    def _add_role_common_impl(self, login, subject_type, role, fields):
        fields = fields or {}

        role_name = role.get('role')
        if role_name not in get_all_permissions():
            raise RoleNotFound('Role does not exist: %s' % role_name)

        user = self._get_user(login, subject_type)

        if role['group'] == 'common':
            add_perms_by_role(role_name, user)
        elif role['group'] == 'system':
            # Пока нам интересно только одно поле scope
            try:
                system = System.objects.get(slug=role['system_on'])
            except System.DoesNotExist:
                raise RoleNotFound('Role system does not exist: %s' % role['system_on'])
            else:
                scope = fields.get('scope') or '/'

                add_perms_by_role(role_name, user, system, scope, check_responsibilities=True)

                return {'data': {'scope': scope}}
        else:
            raise RoleNotFound('Role group not exist: %s' % role['group'])

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        return self._remove_role_common_impl(login, USER_TYPES.USER, role, data, is_fired)

    def remove_tvm_role_impl(self, login, role, data, is_fired, **kwargs):
        return self._remove_role_common_impl(login, USER_TYPES.TVM_APP, role, data, is_fired)

    def _remove_role_common_impl(self, login, subject_type, role, data, is_fired):
        role_name = role.get('role')
        if not role_name:
            raise RoleNotFound('Нет такой роли')

        user = self._get_user(login, subject_type)

        if role['group'] == 'common':
            remove_perms_by_role(role_name, user)
        elif role['group'] == 'system':
            try:
                system = System.objects.get(slug=role['system_on'])
            except System.DoesNotExist:
                pass
            else:
                scope = data.get('scope') or '/'

                remove_perms_by_role(role_name, user, system, scope)
        else:
            raise RoleNotFound('Role does not exist: %s' % role_name)

    def get_roles(self, request):
        """Этот метод можно определить изменением следующих полей:
        GET_ROLES_PAGE_SIZE, GET_ROLES_STREAMS, GET_ROLES_FORM_CLASS
        Список стримов должен быть непуст.
        """
        # Копия базового класса, но исправлена логика получения maxpk, установки exhausted и получения qs
        # функция values_list() принимает набор pk объекто, которые нужно выбрать, а не queryset как в базовом классе
        if not self.GET_ROLES_STREAMS:
            return NotImplemented

        next_url = None
        form = self.GET_ROLES_FORM_CLASS(request.GET, self)
        if not form.is_valid():
            form.raise_first()
        stream_name, since = form.get_clean_data()
        stream_num, stream = self.get_stream_by_name(stream_name, self.GET_ROLES_STREAMS)

        pks = stream.get_queryset()
        maxpk = pks.last()
        if since:
            pks = pks.filter(pk__gt=since)
        pks = pks[:self.GET_ROLES_PAGE_SIZE]
        qs = stream.values_list(pks)
        exhausted = maxpk is None or maxpk in pks
        roles = []
        pk = None
        for row in qs:
            pk = row[0]
            if pk == maxpk:
                exhausted = True
            roles.append(stream.row_as_dict(row))
        page = {
            'code': 0,
            'roles': roles,
        }
        if exhausted:
            if stream_num < len(self.GET_ROLES_STREAMS) - 1:
                next_url = '%s?type=%s' % (request.path, self.GET_ROLES_STREAMS[stream_num + 1].name)
        else:
            next_url = '%s?type=%s&since=%d' % (request.path, stream.name, pk)
        if next_url:
            page['next-url'] = next_url
        return page
