# coding: utf-8


from itertools import chain

import attr
from django.utils.translation import ugettext as _

from idm.core.constants.approverequest import APPROVEREQUEST_DECISION
from idm.core.constants.role import ROLE_STATE
from idm.core.constants.workflow import RUN_REASON, REQUEST_TYPE
from idm.core.workflow.exceptions import AccessDenied
from idm.core.workflow.common.subject import subjectify
from idm.framework.requester import requesterify


@attr.s
class Result(object):
    result = attr.ib()
    message = attr.ib(default='')
    data = attr.ib(default=None)

    def __bool__(self):
        return self.result

    # py2 compat
    __nonzero__ = __bool__


def _check_deprive_possibility(requester, role, to_replace=False, pass_check_state=False, for_api=False):
    """
    Проверки возможности отзыва которые идут до проверок на права доступа
    :returns: False - отзыв не возможен
              None - отзыв возможен, требуется дальнейшая проверка прав доступа
              True - отзыв возможен, безусловно
    """
    requester_user = requester.impersonated
    requester_user_is_idm_robot = requester_user is None or requester_user.is_idm_robot()
    valid_states = ROLE_STATE.CAN_BE_DEPRIVING_API_STATES if for_api else ROLE_STATE.CAN_BE_DEPRIVING_STATES

    if not role.system.is_operational():
        return Result(
            False,
            _('Отзыв невозможен: система сломана или неактивна'))

    if requester_user_is_idm_robot and role.state == ROLE_STATE.DEPRIVING_VALIDATION:
        return Result(True)

    if not (role.state in valid_states or pass_check_state):
        return Result(
            False,
            _('Роль находится в состоянии %s, из которого отзыв невозможен') % role.get_state_display())

    if role.parent_id and not to_replace and not pass_check_state:
        if requester_user_is_idm_robot and not role.parent.is_active:
            return True, ''
        role.fetch_user()
        if role.user and not role.user.is_active:
            if requester_user_is_idm_robot:
                return True, ''
        elif role.state != 'onhold' and role.parent.is_active:
            return Result(
                False,
                _('Нельзя отозвать роль, имеющую родительскую роль, если она не отложена'))

    if requester_user.is_idm_robot():
        return Result(True)
    if requester_user.has_perm('core.deprive_role_without_workflow'):
        return Result(True)
    if role.system.is_permitted_for(requester_user, 'core.deprive_role_without_workflow'):
        if _check_scope(requester_user, role.system, 'core.deprive_role_without_workflow', role.node):
            return Result(True)

    return None


def _check_permission_no_workflow(requester, role):
    """
    Проверка прав пользователя requester для случая если workflow на отзыв не используется
    """
    requester_user = requester.impersonated

    if not requester_user.is_active:
        return Result(False, _('Отзывающий уволен'))

    if not requester.is_allowed_for(role.system):
        return Result(
            False,
            _('Отзыв невозможен: недостаточно прав на исполнение роли в этой системе'))

    if role.last_request_id and role.last_request.requester_id == requester_user.id:
        return Result(True)

    subject = subjectify(role)

    if subject == subjectify(requester_user):
        return Result(True)

    if subject.is_managed_by(requester_user):
        return Result(True)

    if requester_user.has_perm('core.deprive_role'):
        return Result(True)

    if role.system.is_permitted_for(requester_user, 'core.deprive_role'):
        return Result(
            _check_scope(requester_user, role.system, 'core.deprive_role', role.node),
            _('Недостаточно прав для отзыва роли для этого узла'))

    return Result(False, _('Недостаточно прав для отзыва роли'))


def _check_permission_with_workflow(requester, role):
    requester_user = requester.impersonated
    role.system.fetch_actual_workflow()
    try:
        context = role.apply_workflow(
            requester_user, save=False,
            reason=RUN_REASON.DEPRIVE, request_type=REQUEST_TYPE.DEPRIVE)
    except AccessDenied:
        return Result(False, message=_('Недостаточно прав для отзыва роли'))
    approvers = list(chain(*context['approvers']))  # flatten nested lists
    if not approvers or subjectify(requester_user) in (subjectify(a.user) for a in approvers):
        return Result(True)
    return Result(
        False,
        message=_('Недостаточно прав для отзыва роли'),
        data={'approvers': [
            {'full_name': user.get_full_name(),
             'username': user.username,
             'is_active': user.is_active}
            for user in (a.user for a in approvers) if user.is_active]})


def can_request_role(requester, subject, system, node=None, role_fields=None, parent_id=None):
    """Проверяет, может ли requester запросить роль для subject (пользователя или группы) в system.
    role_data и role_fileds - информация о роли
    """

    requester = requesterify(requester)
    requester_user = requester.impersonated

    if not system.is_operational():
        return Result(False, _('Система сломана или неактивна'))

    if not requester.is_allowed_for(system):
        return Result(False, _('Недостаточно прав на исполнение роли в этой системе'))

    subject = subjectify(subject)

    if parent_id is not None:
        if requester_user is None or requester_user.is_idm_robot():
            return Result(True)
        return Result(False, _('Только роботы могут запрашивать связанные роли'))

    if not subject.is_requestable_for():
        return Result(False, _('Владелец уволен или группа удалена'))

    if system.request_policy == 'anyone':
        return Result(True)
    elif system.request_policy == 'subordinates':
        if subjectify(requester_user) == subject:
            return Result(True)

        if subject.is_user:
            subject.user.fetch_department_group()

        if subject.is_managed_by(requester_user):
            return Result(True)

        if requester_user.has_perm('core.request_role'):
            return Result(True)

        if requester_user.is_idm_robot():
            return Result(True)

        if system.is_permitted_for(requester_user, 'core.request_role'):
            if node:
                return Result(
                    _check_scope(requester_user, system, 'core.request_role', node),
                    _('Недостаточно прав для запроса роли на этот узел')
                )
            else:
                return Result(True)

    return Result(False, _('Недостаточно прав'))


def can_deprive_role(requester, role, to_replace=False, pass_check_state=False, for_api=False):
    """Проверяет, может ли requester отозвать role"""

    requester = requesterify(requester)
    result = _check_deprive_possibility(requester, role, to_replace=to_replace,
                                        pass_check_state=pass_check_state, for_api=for_api)
    if result is not None:
        return result

    if role.system.use_workflow_for_deprive:
        if for_api:
            return Result(True)
        return _check_permission_with_workflow(requester, role)
    else:
        return _check_permission_no_workflow(requester, role)


def can_approve_role(user, role):
    """Проверяет, может ли пользователь user подтвердить или отклонить роль role"""

    requester = requesterify(user)
    requester_user = requester.impersonated

    from idm.core.models import ApproveRequest

    if role.state not in ROLE_STATE.REQUESTED_STATES:
        return Result(False, _('Невозможно подтвердить роль: состояние роли не требует подтверждения'))

    if not requester_user.is_active:
        return Result(False, _('Невозможно подтвердить роль: подтверждающий уволен'))

    user_approve_requests = ApproveRequest.objects.filter(approve__role_request__role=role, approver=requester_user)
    if not user_approve_requests.exists():
        return Result(False, _('Невозможно подтвердить роль: недостаточно прав'))
    elif user_approve_requests.filter(approve__approved=None, decision=APPROVEREQUEST_DECISION.NOT_DECIDED).exists():
        return Result(True)
    else:
        return Result(False, _('Невозможно подтвердить роль: решение уже принято'))


def can_retry_failed_role(user, role):
    """Проверяет, может ли пользователь user довыдать роль role"""
    requester = requesterify(user)
    requester_user = requester.impersonated

    if not role.system.is_operational():
        return False

    if not requester.is_allowed_for(role.system):
        return Result(
            False,
            _('Недостаточно прав на исполнение роли в этой системе'),
        )

    if not (role.is_personal_role() and role.state == ROLE_STATE.FAILED or
            role.is_group_role() and role.get_personal_roles().filter(state=ROLE_STATE.FAILED).count() > 0):
        return Result(
            False,
            _(
                'Довыдать можно только персональную роль или групповую роль, у которой есть персональные со статусом \'Ошибка\''),
        )

    subject = subjectify(role)

    if subject == subjectify(requester_user):
        return Result(True)

    if subject.is_managed_by(requester_user):
        return Result(True)

    if requester_user.has_perm('core.retry_failed_role'):
        return Result(True)

    if requester_user.is_idm_robot():
        return Result(True)

    if role.system.is_permitted_for(requester_user, 'core.retry_failed_role'):
        return Result(
            _check_scope(requester_user, role.system, 'core.retry_failed_role', role.node),
            _('Недостаточно прав для довыдачи роли для этого узла'),
        )

    return Result(False, _('Недостаточно прав для довыдачи роли'))


def can_rerequest_role(user: 'User', role: 'Role') -> Result:  # noqa
    """Проверяет, может ли пользователь user перезапросить роль role"""

    requester = requesterify(user)

    if not role.system.is_operational():
        return Result(False, _('Система неактивна или сломана'))

    if role.has_expired_ttl():
        return Result(False, _('Нельзя перезапросить временную роль. Запросите новую роль через форму пожалуйста.'))

    # пользователь должен иметь право запросить такую же роль
    subject = subjectify(role)
    result = can_request_role(requester, subject, role.system, role.node, role.fields_data, role.parent_id)
    if result:
        if role.state in ROLE_STATE.REREQUESTABLE_STATES or (
                role.state in ROLE_STATE.AUTHORITY_REREQUESTABLE_STATES and
                requester.impersonated.has_perm('core.rerequest_granted_role')
        ):
            return Result(True)
        result = Result(False, _('Нельзя перезапросить роль в состоянии "%s"') % role.get_state_display())

    return result


def _check_scope(user, system, permission, node):
    from idm.core.models import RoleNode
    closures = RoleNode.objects.get_permitted_query(user, permission, system)
    return closures.filter(child_id=node.id).exists()
