# coding: utf-8


import logging

from django.apps import apps
from django.conf import settings
from django.contrib.auth.management import create_permissions
from django.contrib.auth.models import Permission
from django.db import connection
from django_idm_api.exceptions import RoleNotFound, AccessDenied
from guardian.shortcuts import assign_perm, remove_perm, get_users_with_perms

log = logging.getLogger(__name__)


_permissions = None


def get_permission(permission):
    global _permissions
    if _permissions is None:
        permissions = Permission.objects.select_related('content_type')
        perms = {(perm.content_type.app_label, perm.codename): perm for perm in permissions}
        _permissions = perms
    app_label, codename = permission.split('.', 1)
    result = _permissions.get((app_label, codename))
    if result is None:
        raise Permission.DoesNotExist('No such permission')
    return result


def clear_permission_cache():
    global _permissions
    _permissions = None


def get_all_permissions():
    return dict(settings.IDM_COMMON_ROLES_PERMISSIONS, **settings.IDM_SYSTEM_ROLES_PERMISSIONS)


def get_permissions_for_role(role):
    return get_all_permissions().get(role, [])


def get_or_create_internal_role(**kwargs):
    from idm.core.models import InternalRole

    internal_role, _ = InternalRole.objects.get_or_create(**kwargs)
    return internal_role


def add_perms_by_role(role_name, user, system=None, scope='/', check_responsibilities=False):
    """Выдает внутреннюю роль пользователю
    """
    from idm.core.models import InternalRole, RoleNode

    assert role_name in get_all_permissions()

    if system is None:
        internal_role, _ = InternalRole.objects.get_or_create(role=role_name)
        assign_perms_for_internal_role(internal_role, user)

        if role_name not in settings.IDM_NO_STAFF_INTERNAL_ROLES:
            user.is_staff = True
        if role_name == 'superuser':
            user.is_superuser = True  # для админки
        user.save(update_fields=('is_staff', 'is_superuser'))
    else:
        # Добавляем InternalRole
        if isinstance(scope, RoleNode):
            node = scope
        else:
            try:
                node = RoleNode.objects.get_node_by_value_path(system, scope)
            except RoleNode.DoesNotExist:
                raise RoleNotFound('Role `%s` scope in system `%s` does not exist: %s' % (role_name, system.slug,
                                                                                          scope))
        if check_responsibilities and node.responsibilities.filter(is_active=True).exists():
            raise AccessDenied('Scope `%s` in system `%s` is managed by responsibilities' % (scope, system.slug))
        internal_role = get_or_create_internal_role(role=role_name, node=node)
        assign_perms_for_internal_role(internal_role, user)
    user.drop_permissions_cache()


def remove_perms_by_role(role_name, user, system=None, scope='/'):
    """Забирает внутреннюю роль у пользователя
    """

    from idm.core.models import InternalRole, RoleNode
    if role_name not in get_all_permissions():
        return

    if system is None:
        internal_roles = InternalRole.objects.filter(role=role_name)
        if not internal_roles.exists():
            return
        else:
            for internal_role in internal_roles:
                remove_perms_for_internal_role(internal_role, user)

            # Если у юзера не осталось внутренних ролей, снимаем флаг is_staff
            # при этом некоторые роли флаг is_staff обеспечивать не должны
            if not InternalRole.objects.filter(
                    user_object_permissions__user=user, node=None
            ).exclude(
                role__in=settings.IDM_NO_STAFF_INTERNAL_ROLES
            ).exists():
                user.is_staff = False

            if role_name == 'superuser':
                user.is_superuser = False
            user.save(update_fields=('is_staff', 'is_superuser'))
    else:
        # Отзываем права и удаляем InternalRole
        if isinstance(scope, RoleNode):
            nodes = [scope]
        else:
            nodes = RoleNode.objects.get_all_nodes_by_value_path(system, scope)

            if not nodes.exists():
                raise RoleNotFound('Role `%s` scope in system `%s` does not exist: %s' % (role_name, system.slug,
                                                                                          scope))
        internal_roles = InternalRole.objects.filter(role=role_name, node__in=nodes)
        if not internal_roles.exists():
            raise RoleNotFound('Role `%s` does not exist for given nodes' % (role_name))

        for internal_role in internal_roles:
            remove_perms_for_internal_role(internal_role, user)

    for internal_role in internal_roles:
        if not get_users_with_perms(internal_role, with_group_users=False).exists():
            internal_role.delete()

    user.drop_permissions_cache()


def assign_perms_for_internal_role(role, user):
    for perm in get_permissions_for_role(role.role):
        log.info('assign permission %s on node `%s` to user %s',
                 perm, role.node_id, user.username)

        assign_perm('core.%s' % perm, user, role)


def remove_perms_for_internal_role(role, user):
    for perm in get_permissions_for_role(role.role):
        log.info('remove permission %s on node `%s` to user %s',
                 perm, role.node_id, user.username)

        remove_perm('core.%s' % perm, user, role)


def local_or_global(qs, local_only=False, global_only=False, root_only=False):
    if global_only:
        qs = qs.global_()
    elif local_only:
        qs = qs.local()
    elif root_only:
        qs = qs.root()
    return qs


def sync_permissions(add_only=False, remove_only=False, global_only=False, local_only=False, root_only=False):
    global _permissions
    from idm.core.models import InternalRole, InternalRoleUserObjectPermission
    from idm.users.models import User

    for app_config in apps.get_app_configs():
        create_permissions(app_config, verbosity=1, interactive=False)

    if not remove_only:
        for user in User.objects.all():
            qs = InternalRole.objects.filter(user_object_permissions__user=user)
            qs = local_or_global(qs, local_only=local_only, global_only=global_only, root_only=root_only)
            qs = qs.distinct()
            for internal_role in qs:
                assign_perms_for_internal_role(internal_role, user)

    if not add_only:
        for role, permissions in get_all_permissions().items():
            qs = (
                InternalRoleUserObjectPermission.objects.
                    filter(content_object__role=role).
                    exclude(permission__codename__in=permissions)
            )
            qs = local_or_global(qs, local_only=local_only, global_only=global_only, root_only=root_only)
            qs.delete()
    _permissions = None
