# coding: utf-8

"""
Ping management command.
Prints '0;OK' if all ok.
Prints '2;{error description}' in case of error.
"""
import sys

from django.conf import settings
from django.core.management.base import BaseCommand
from django.db import DEFAULT_DB_ALIAS
from django_replicated.dbchecker import db_is_alive, db_is_writable


# all green
OK = '0;OK'


class DatabaseError(Exception):
    """
    Некоторые из инстансов базы недоступны.
    """


class CriticalDatabaseError(DatabaseError):
    """
    Недоступен мастер или мастер находится в read-only
    """


def check_databases(aliases):
    """
    Бросить исключение, если некоторые из инстансов базы недоступны.

    @type aliases: list
    @param aliases: список имен алиасов из settings.DATABASES
    @rtype: None
    """
    unavailable_aliases = []

    for alias in aliases:
        if not db_is_alive(alias, number_of_tries=3):
            unavailable_aliases.append(alias)

    message = ''
    if unavailable_aliases:
        message = 'Hosts {} are unavailable'.format(','.join(unavailable_aliases))
        if DEFAULT_DB_ALIAS in unavailable_aliases:
            # мастер недоступен
            raise CriticalDatabaseError(message)

    if not db_is_writable(DEFAULT_DB_ALIAS):
        raise CriticalDatabaseError('Master is in read-only state')

    if unavailable_aliases:
        raise DatabaseError(message)


def format_error(message, warning=False):
    code = '1' if warning else '2'
    return '{0};{1}'.format(code, message)


class Command(BaseCommand):
    help = 'Ping service and return status in monrun-compliant format'
    requires_system_checks = False

    def handle(self, *args, **options):
        stdout = sys.stdout
        if options.get('verbosity') not in ['2', '3']:
            # RULES-1256
            # raven-python при завершении процесса может написать в stdout свои сообщения,
            # после чего monrun не сможет правильно распарсить вывод.
            # Подменяем sys.stdout на sys.stderr, который вырезается в monrun-конфиге.
            sys.stdout = sys.stderr

        try:
            check_databases(list(settings.DATABASES.keys()))
        except CriticalDatabaseError as exc:
            stdout.write(format_error('Database error:"{0}"'.format(exc)))
        except DatabaseError as exc:
            stdout.write(format_error('Database error:"{0}"'.format(exc), warning=True))
        except Exception as exc:
            stdout.write(format_error('Unhandled exception:"{0}"'.format(repr(exc))))

        else:
            stdout.write(OK)

        stdout.write('\n')
