# coding: utf-8


from collections import defaultdict, deque

from django.db import models
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from idm.framework.mixins import LocalizedModel, DifferenceTableMixin
from idm.nodes.models import ExternalTrackingNode
from idm.services import canonical
from idm.services.fetchers import IDSServiceFetcher
from idm.services.managers import ServiceManager
from idm.services.queues import ServiceQueue


class Service(LocalizedModel, ExternalTrackingNode, DifferenceTableMixin):
    external_state = models.CharField(_('Состояние'), max_length=255, default='unknown_state', db_index=True)

    created_at = models.DateTimeField(_('Дата создания'), default=timezone.now, editable=False, null=True)
    updated_at = models.DateTimeField(_('Дата обновления'), editable=False, null=True)

    external_id = models.IntegerField(verbose_name=_('Внешний ID'), default=0, unique=True)

    is_vteam = models.BooleanField(default=False, verbose_name=_('VTeam'))

    name = models.CharField(_('Имя сервиса'), max_length=255, default='')
    name_en = models.CharField(_('Имя сервиса (на английском)'), max_length=255, default='')
    description_wiki = models.TextField(blank=True, default='', verbose_name=_('Описание (wiki)'))
    description_wiki_en = models.TextField(blank=True, default='', verbose_name=_('Описание (wiki, на английском)'))
    description_html = models.TextField(blank=True, default='', verbose_name=_('Описание (HTML)'))
    description_html_en = models.TextField(blank=True, default='', verbose_name=_('Описание (HTML, на английском)'))

    root = models.ForeignKey('self', null=True, blank=True, verbose_name=_('Корневой сервис'), on_delete=models.CASCADE)
    membership_inheritance = models.BooleanField(default=True, verbose_name=_('Нужно ли наследовать роли предков'))

    objects = ServiceManager()
    fetcher = IDSServiceFetcher()
    queue_class = ServiceQueue
    EXTERNAL_ID_FIELD = 'external_id'
    REQUIRE_EXTERNAL_ID = True
    SUPPORTS_RESTORE = True

    class Meta:
        verbose_name = _('Сервис')
        verbose_name_plural = _('Сервисы')

    def __str__(self):
        if self.root_id:
            return '{root} / {name} ({state})'.format(root=self.root.name, name=self.name, state=self.state)
        else:
            return '{name} ({state})'.format(name=self.name, state=self.state)

    def get_name(self, lang=None):
        return self.get_localized_field('name', lang=lang)

    def as_canonical(self):
        return canonical.CanonicalService(
            hash=self.hash,
            is_vteam=self.is_vteam,
            state=self.external_state,
            slug=self.slug.lower(),
            name=self.name,
            name_en=self.name_en,
            description_wiki=self.description_wiki,
            description_wiki_en=self.description_wiki_en,
            description_html=self.description_html,
            description_html_en=self.description_html_en,
            external_id=self.external_id,
            parent_id=self.parent.external_id if self.parent else None,
            root_id=self.root.external_id if self.root else None,
            membership_inheritance=self.membership_inheritance,
            children=(),
        )

    @classmethod
    def from_canonical(cls, canonical):
        obj = cls()
        obj.update_from_canonical(canonical)
        return obj

    def update_from_canonical(self, canonical):
        different_fields = {
            'state': 'external_state',
        }

        fields_to_ignore = [
            'children',
            'parent_id',
            'root_id',
        ]

        self.root = Service.objects.get_by_external_id(canonical.root_id)

        for canonical_attribute in canonical.__attrs_attrs__:
            canonical_name = canonical_attribute.name
            if canonical_name in different_fields:
                actual_name = different_fields[canonical_name]
            else:
                actual_name = canonical_name

            if actual_name not in fields_to_ignore:
                value = getattr(canonical, canonical_name)
                setattr(self, actual_name, value)

    def save(self, *args, **kwargs):
        now = timezone.now()
        if self.pk is None:
            self.created_at = now
        self.updated_at = now
        return super(Service, self).save(*args, **kwargs)

    def synchronize(self):
        result, queue = super(Service, self).synchronize()
        refreshed = type(self).objects.get(pk=self.pk)
        refreshed.save(update_fields=('updated_at',))
        return result, queue



