# coding: utf-8


from idm.nodes.queue import (ExternalIDQueue, ExternalIDAdditionItem, ExternalIDMovementItem, ModificationItem,
                             RestoringItem, ExternalIDRemovalItem)


class ServiceAdditionItem(ExternalIDAdditionItem):
    def apply(self, **kwargs):
        from idm.services.models import Service
        service = Service.from_canonical(self.child_data)
        service.parent = self.get_parent_node()
        service.save()
        self.new_node = service
        service.actions.create(action='service_created')


class ServiceRestoringItem(RestoringItem):
    def apply(self, **kwargs):
        self.node.state = 'active'
        self.node.external_state = self.extra['data'].state
        self.node.save(update_fields=['state', 'external_state'])
        self.node.actions.create(action='service_restored')


class ServiceModificationItem(ModificationItem):
    def apply(self, **kwargs):
        diff = self.get_diff()
        data_has_changed = bool(diff.flat)

        self.node.update_from_canonical(self.new_data)
        self.node.hash = self.new_data.hash
        self.node.save()
        if data_has_changed:
            self.node.actions.create(
                service=self.node,
                action='service_changed',
                data={
                    'diff': {item.name: [item.old, item.new] for item in diff.flat}
                }
            )
        return self.node


class ServiceMovementItem(ExternalIDMovementItem):
    def apply(self, **kwargs):
        new_parent = self.get_parent_node()

        action_data = {
            'moved_from': {
                'id': self.node.pk,
                'path': self.node.path,
            }
        }

        self.node.parent = new_parent
        self.node.save(recalc_fields=True)

        action_data['moved_to'] = self.node.path
        self.node.actions.create(action='service_moved', data=action_data)


class ServiceRemovalItem(ExternalIDRemovalItem):
    def apply(self, **kwargs):
        self.node.state = 'depriving'
        self.node.external_state = 'deleted'
        self.node.save(update_fields=['state', 'external_state'])
        self.node.actions.create(action='service_removed')


class ServiceQueue(ExternalIDQueue):
    addition = ServiceAdditionItem
    modification = ServiceModificationItem
    movement = ServiceMovementItem
    restore = ServiceRestoringItem
    removal = ServiceRemovalItem

    def get_item_class(self):
        from idm.services.models import Service
        return Service
