# coding: utf-8


import logging

import attr
import ldap
from attr.validators import instance_of
from django.conf import settings
from django.utils.encoding import force_bytes, force_text

log = logging.getLogger(__name__)


@attr.s
class AccountManager(object):
    active_users_ou_list = settings.AD_ACTIVE_USERS_OU
    old_users_ou = settings.AD_LDAP_OLD_USERS_OU

    connector = attr.ib()

    def search(self, username):
        """
        Ищет пользователя по логину среди Users, ForeignUsers и OldUser.
        """
        log.info('Searching user %s', username)
        account = None
        for org_unit in self.active_users_ou_list + [self.old_users_ou]:
            account_data = self.connector.search_s(
                org_unit,
                ldap.SCOPE_SUBTREE,
                '(SAMAccountName=%s)' % username
            )
            if account_data:
                log.info('User %s was found in %s', username, org_unit)
                account = Account.from_ldap(
                    org_unit=org_unit,
                    username=username,
                    account_data=account_data,
                    connector=self.connector,
                )
                return account
        log.info('User %s was not found in AD', username)
        return account


@attr.s
class Account(object):
    old_users_ou = settings.AD_LDAP_OLD_USERS_OU

    connector = attr.ib()
    org_unit = attr.ib(validator=instance_of(str))
    username = attr.ib(validator=instance_of(str))
    display_name = attr.ib(validator=instance_of(str))
    properties = attr.ib(validator=instance_of(dict))

    @classmethod
    def from_ldap(cls, org_unit, account_data, username, connector=None):
        if not isinstance(account_data, list):
            raise TypeError('Wrong account data')
        if len(account_data) != 1:
            raise TypeError('Wrong account data len')
        account_data = account_data[0]
        if len(account_data) != 2:
            raise TypeError('Wrong account data sublen')
        display_name, properties = account_data
        return cls(
            org_unit=org_unit,
            display_name=display_name,
            username=username,
            properties=properties,
            connector=connector,
        )

    def has_effective_groups(self):
        return bool(self.get_effective_groups())

    def is_fully_blocked(self):
        """Чтобы считаться полностью заблокированным, аккаунт должен:
        * иметь признак блокировки
        * находиться в Old Users
        """
        result = not self.is_active() and self.is_old()
        return result

    def is_active(self):
        account_control = int(self.properties['userAccountControl'][0])
        result = account_control & settings.ADS_UF_ACCOUNTDISABLE == 0
        return result

    def is_old(self):
        return self.org_unit.lower() == self.old_users_ou.lower()

    def disable(self):
        """
        Выставить в AD флаг ADS_UF_ACCOUNTDISABLE
        """
        account_control = int(self.properties['userAccountControl'][0])
        diff = [(
            ldap.MOD_REPLACE,
            'userAccountControl', force_bytes(account_control | settings.ADS_UF_ACCOUNTDISABLE))
        ]
        self.connector.modify_s(force_text(self.display_name), diff)

    def enable(self):
        """
        Снять флаг ADS_UF_ACCOUNTDISABLE в AD
        """
        account_control = int(self.properties['userAccountControl'][0])
        # FIXME проверить
        if account_control == account_control | settings.ADS_UF_ACCOUNTDISABLE:  # если заблокирован
            diff = [
                (ldap.MOD_REPLACE, 'userAccountControl', force_bytes(account_control ^ settings.ADS_UF_ACCOUNTDISABLE))
            ]
            self.connector.modify_s(force_text(self.display_name), diff)

    def add_to_group(self, group_dn):
        """
        Добавить учетку в группу в AD
        """

        user_groups = set(self.properties.get('memberOf', []))
        log.info('Adding user %s to group %s', self.username, group_dn)

        if group_dn in user_groups:
            log.warning('User %s already in group %s', self.username, group_dn)
            return False

        diff = [(ldap.MOD_ADD, 'member', force_bytes(self.display_name))]
        self.connector.modify_s(force_text(group_dn), diff)

    def get_effective_groups(self):
        user_groups = set(self.properties.get('memberOf', []))
        ignore_groups = set(settings.AD_IGNORE_GROUPS)
        return user_groups - ignore_groups

    def remove_from_groups(self):
        """
        Удалить учетку пользователя из групп
        """

        removed_all = True
        for group_dn in self.get_effective_groups():
            log.info('Removing user %s from group %s', self.username, group_dn)
            try:
                self.remove_from_group(group_dn)
            except Exception:
                log.exception('Cannot remove user %s from group %s', self.username, group_dn)
                removed_all = False
        return removed_all

    def remove_from_group(self, group_dn):
        """
        Удалить учетку из группы в AD
        """

        log.info('Removing user %s from group %s', self.username, group_dn)
        groups = set(self.properties.get('memberOf', []))
        if group_dn not in groups:
            log.warning('user %s is not a member in group %s', self.username, group_dn)
            return False

        diff = [(ldap.MOD_DELETE, 'member', force_bytes(self.display_name))]
        self.connector.modify_s(force_text(group_dn), diff)

    def move_to(self, move_to):
        if self.display_name.endswith(move_to):
            log.info('User %s already in %s', self.username, move_to)
            return

        self.connector.rename_s(self.display_name, 'CN=' + force_text(self.properties['cn'][0]), force_text(move_to))
        self.properties['cn'][0] = move_to

    def move_to_old(self):
        """
        Перенести учетку в OldUsers
        """
        self.move_to(self.old_users_ou)
