# -*- coding: utf-8 -*-

import logging
import socket

from blackbox import Blackbox
from django.conf import settings

from idm.utils import http


REGISTRATION_URL = settings.PASSPORT_URL + '/1/bundle/account/register/?consumer=%s' % settings.PASSPORT_CONSUMER
SUBSCRIPTION_URL = settings.PASSPORT_URL + '/1/account/%%s/subscription/strongpwd/?consumer=%s' % settings.PASSPORT_CONSUMER
log = logging.getLogger(__name__)


HAVE_PASSWORD_ATTRIBUTE = '1005'


class PassportError(Exception):
    pass


class UnknownLoginError(PassportError):
    pass


class NoGrantsError(PassportError):
    pass


def get_uid(login: str) -> str:
    try:
        blackbox = get_external_blackbox()
        return blackbox.uid(login)

    except Exception:
        log.exception('error during checking "%s" in blackbox', login)
        raise


def exists(login: str) -> bool:
    # лезем проверять паспортный логин в blackbox
    return get_uid(login) is not None


def get_external_blackbox():
    blackbox = getattr(settings, 'IDM_EXTERNAL_BLACKBOX_INSTANCE', None)
    if blackbox is None:
        blackbox = Blackbox(
            settings.BLACKBOX_PASSPORT_URL,
            tvm2_client_id=settings.TVM2_CLIENT_ID,
            tvm2_secret=settings.TVM2_SECRET,
            blackbox_client=settings.BLACKBOX_CLIENT
        )
    return blackbox


def register_login(login, username, first_name, last_name, subscribe, retries=3):
    """
    https://wiki.yandex-team.ru/passport/python/api/bundle/registration/#registracijapolzovateljabezpodtverzhdenija
    :param login:
    :param username:
    :param first_name:
    :param last_name:
    :param subscribe:
    :param retries:
    :return: True
    """
    log.info('Register passport login "%s" for "%s"', login, username)
    log.debug('GET %s', REGISTRATION_URL)

    assert login.startswith(f'yndx-{username}'), f'Trying to register login {login} for other user {username}'

    data = {
        'login': login,
        'ignore_stoplist': 'true',
        'firstname': first_name,
        'lastname': last_name,
        'email': settings.PASSPORT_EMAIL.format(username=username),
    }
    if subscribe:
        data['subscriptions'] = 'strongpwd'
    headers = {
        'Ya-Client-User-Agent': settings.IDM_USERAGENT,
        'Ya-Consumer-Client-Ip': _get_current_ip(),  # У нас нет ip клиента
    }

    try:
        response = http.post(
            REGISTRATION_URL,
            data=data,
            headers=headers,
            tvm_id=settings.PASSPORT_TVM_ID,
        )

    except Exception:
        log.exception('Cannot register login "%s" for "%s"', login, username)
        raise

    # Важно. Ответ "account.already_registered" считается ошибкой, потому что сюда попадают только несуществующие логины
    if response.status_code == 200:
        try:
            answer = response.json()

            if answer['status'] == 'ok':
                log.info('Login <%s> registered', login)
                return True

            else:
                log.warning('Bad status %s from passport for "%s": %s', answer['status'], username, response.content)

                if 'backend.blackbox_failed' in answer['errors'] or 'backend.database_failed' in answer['errors']:
                    if retries > 0:
                        return register_login(login, username, first_name, last_name, retries=retries-1)
                    else:
                        log.warning('Out of retries, give up for "%s": %s', username, response.content)

        except Exception:
            log.warning('Bad answer from %s for %s: %s', REGISTRATION_URL, username, response.content)

    elif response.status_code == 403:
        log.error('No enough grants to register login %s: %s', username, response.content)
        raise NoGrantsError('No enough grants to register login %s: %s' % (username, response.content))

    else:
        log.warning('Unknown http answer (%s): %s', response.status_code, response.content)

    raise PassportError('Cannot register login')


def _get_current_ip():
    """
        > socket.getaddrinfo("www.python.org", 80, proto=socket.SOL_TCP)
        [(2, 1, 6, '', ('82.94.164.162', 80)),
         (10, 1, 6, '', ('2001:888:2000:d::a2', 80, 0, 0))
        :return: ip address
    """
    struct = socket.getaddrinfo(socket.gethostname(), 443)
    return struct[0][4][0]


def set_strongpwd(login):
    """
    Устанавливает паспортному логину усиленную парольную политику sid67 (strongpwd)
    https://beta.wiki.yandex-team.ru/passport/api/bundle/managesids/
    :param login: паспортный логин
    :return: bool - успешность операции
    """
    login_uid = get_uid(login)

    if not login_uid:
        raise UnknownLoginError('Unknown login %s' % login)

    subscribe_url = SUBSCRIPTION_URL % login_uid
    log.debug('GET %s', subscribe_url)

    try:
        log.info('Subscribing login "%s" (%s)', login, login_uid)
        response = http.post(subscribe_url, data={}, tvm_id=settings.PASSPORT_TVM_ID)

    except Exception:
        log.exception('Cannot subscribe login "%s"', login)
        raise

    if response.status_code == 200:
        log.info('Login "%s" (%s) subscribed', login, login_uid)
        return True

    else:
        log.error('Bad answer from %s for %s: %s', subscribe_url, login, response.text)
        return False


def remove_strongpwd(login):
    """
    Снимает паспортному логину усиленную парольную политику sid67 (strongpwd)
    https://beta.wiki.yandex-team.ru/passport/api/bundle/managesids/
    :param login: паспортный логин
    :return: bool - успешность операции
    """
    login_uid = get_uid(login)

    if not login_uid:
        raise UnknownLoginError('Unknown login %s' % login)

    subscribe_url = SUBSCRIPTION_URL % login_uid
    log.debug('DELETE %s', subscribe_url)

    try:
        log.info('Unubscribing login "%s" (%s)', login, login_uid)
        response = http.delete(subscribe_url, tvm_id=settings.PASSPORT_TVM_ID)

    except Exception:
        log.exception('Cannot unsubscribe login "%s"', login)
        raise

    if response.status_code == 200:
        log.info('Login "%s" (%s) unsubscribed', login, login_uid)
        return True

    elif response.status_code == 404:
        log.info('Login "%s" (%s) already unsubscribed', login, login_uid)
        return True

    else:
        log.error('Bad answer from %s for %s: %s', subscribe_url, login, response.text)
        return False


def check_if_fully_registered(login):
    bb = get_external_blackbox()
    info = bb.userinfo(login, userip=None, by_login=True, attributes=HAVE_PASSWORD_ATTRIBUTE)
    have_password = False
    attributes = info.get('attributes')
    if attributes:
        have_password = attributes.get(HAVE_PASSWORD_ATTRIBUTE, False)
    return have_password in ('1', 1)

