# -*- coding: utf-8 -*-
import logging

from django.db import transaction

from idm.core.models import NetworkMacro
from idm.integration import racktables

log = logging.getLogger(__name__)


MACROS_BATCH_SIZE = 1000


def sync_macros(force: bool = False):
    remote = set(racktables.get_macros())

    local_active = set()
    local_inactive = set()
    for slug, is_active in NetworkMacro.objects.values_list('slug', 'is_active').iterator(): # type: str, bool
        if is_active:
            local_active.add(slug)
        else:
            local_inactive.add(slug)

    to_add = sorted(remote - local_active - local_inactive)
    to_restore = sorted((remote - local_active) & local_inactive)
    to_remove = sorted(local_active - remote)

    # TODO: раскомментировать после IDM-7412: Сделать мониторинги на время последнего успешного завершения тасок
    # if not force and len(to_remove) > settings.IDM_RACKTABLES_REMOVED_MACROS_THRESHOLD:
    #     log.error(
    #         'sync_macros: removal threshold violation (%s > %s)',
    #         len(to_remove),
    #         settings.IDM_RACKTABLES_REMOVED_MACROS_THRESHOLD,
    #     )
    #     raise RuntimeError('sync_macros: removal threshold violation')

    for batch_index in range(0, len(to_add), MACROS_BATCH_SIZE):
        with transaction.atomic():
            batch = to_add[batch_index:batch_index+MACROS_BATCH_SIZE]
            for macro in batch:
                try:
                    NetworkMacro.objects.create(slug=macro)
                except Exception:
                    log.exception('Fail while creating macro %s', macro)

    for batch_index in range(0, len(to_restore), MACROS_BATCH_SIZE):
        with transaction.atomic():
            batch = to_restore[batch_index:batch_index+MACROS_BATCH_SIZE]
            try:
                objects = NetworkMacro.objects.filter(slug__in=batch).order_by('slug')
                objects.lock_by_select()
                objects.update(is_active=True)
            except Exception:
                log.exception('Fail while restoring macros')

    for batch_index in range(0, len(to_remove), MACROS_BATCH_SIZE):
        with transaction.atomic():
            batch = to_remove[batch_index:batch_index + MACROS_BATCH_SIZE]
            try:
                objects = NetworkMacro.objects.filter(slug__in=batch).order_by('slug')
                objects.lock_by_select()
                objects.update(is_active=False)
            except Exception:
                log.exception('Fail while removing macros')


