# -*- coding: utf-8 -*-


import logging
import re
import urllib.request, urllib.parse, urllib.error

from django.conf import settings
from django.db import transaction
from django.utils import timezone

from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.utils import update_model
from idm.nodes.fetchers import IDSFetcher
from idm.users.canonical import CanonicalMember
from idm.users.constants.group import GROUP_TYPES
from idm.users.constants.user import USER_TYPES
from idm.users.models import Group, GroupMembership, GroupResponsibility, User
from idm.users.sync.groups import sync_indirect_memberships
from idm.utils import http

log = logging.getLogger(__name__)


USER_FETCH_BATCH_SIZE = 5000

USER_UPDATE_BATCH_SIZE = 1000

STAFF_API_FIELDS = (
    'id', 'uid', 'name.first.ru', 'name.last.ru', 'login', 'personal.gender',
    'work_email', 'official.is_dismissed', 'official.join_at',
    'official.quit_at', 'department_group.id', 'language.ui',
    'name.first.en', 'name.last.en', 'location.office.id',
    'phones.number', 'phones.type', 'official.position.ru',
    'official.is_robot', 'official.affiliation', 'robot_owners.person.id',
)

DEPARTMENT_FIELDS = (
    'id', 'parent', 'name', 'chief', 'url'
)


def parse_date(date):
    return timezone.datetime.strptime(date, '%Y-%m-%d').date() if date else None


def update_user(user_data, user_cache):
    """
    Обновить пользователя данными, полученными из Staff API.
    Если пользователь сменил отдел - изменить его департаментную группу

    @type user_data: dict
    @type user_cache: dict
    @return: need_sync_indirect - bool, флаг означающий нужно ли запустить синхронизацию опосредованных членств
    """
    mobile_phone = None
    if user_data['phones']:
        mobile_phones = [x for x in user_data['phones'] if x.get('type') == 'mobile']
        if mobile_phones:
            mobile_phone = mobile_phones[0].get('number')
            if mobile_phone:
                mobile_phone = mobile_phone.split(',')[0].strip()
                mobile_phone = re.sub('[^+0-9]', '', mobile_phone)

    gender_map = {
        'male': 'M',
        'female': 'F',
    }

    user_fields = {
        'uid': user_data['uid'],
        'username': user_data['login'],
        'email': user_data['work_email'] or '',
        'first_name': user_data['name']['first']['ru'] or '',
        'last_name': user_data['name']['last']['ru'] or '',
        'is_active': not user_data['official']['is_dismissed'],
        'date_joined': parse_date(user_data['official']['join_at']),
        'sex': gender_map[user_data['personal']['gender']],
        'center_id': user_data['id'],
        'staff_id': user_data['id'],
        'fired_at': parse_date(user_data['official']['quit_at']),
        'first_name_en': user_data['name']['first']['en'],
        'last_name_en': user_data['name']['last']['en'],
        'lang_ui': user_data['language']['ui'],
        'mobile_phone': mobile_phone,
        'position': user_data['official']['position']['ru'],
        'is_robot': user_data['official']['is_robot'],
        'affiliation': user_data['official']['affiliation'],
    }

    for namefield in ('first_name', 'last_name', 'first_name_en', 'last_name_en'):
        user_fields[namefield] = user_fields[namefield][:40]

    if not user_data['official']['is_dismissed']:
        # для уволенных ничего не меняем
        user_fields['is_homeworker'] = (user_data['location']['office']['id'] == 14)
        # выставляем флаг активности в LDAP
        # (возможно, это пользователь, который пришел в компанию снова,
        # а значит он снова в активных пользователях в LDAP)
        user_fields['ldap_active'] = True
        user_fields['ldap_blocked_timestamp'] = None
        # Если пользователь не уволен, время увольнения ставим None
        user_fields['idm_found_out_dismissal'] = None

    try:
        user = user_cache[user_data['login']]
        # Если пользователя только что уволили, надо проставить дату, когда IDM узнал про увольнение
        if user.is_active and user_data['official']['is_dismissed']:
            user_fields['idm_found_out_dismissal'] = timezone.now()
    except KeyError:
        user = User.objects.create(
            username=user_data['login'], 
            center_id=user_data['id'],
            notify_responsibles=user_data['official']['is_robot'],
        )
        log.warning('User with login "%s" missing in LDAP', user_data['login'])

    # Проставляем ответственных роботам
    if user_data['official']['is_robot']:
        # иногда robot_owners отсутствует
        owners_data = user_data.get('robot_owners', [])
        new_resp_ids = {user['person']['id'] for user in owners_data}
        old_resp_ids = {u.staff_id for u in user.responsibles.all()}
        if new_resp_ids - old_resp_ids:
            to_add = User.objects.filter(staff_id__in=new_resp_ids - old_resp_ids)
            user.add_responsibles(to_add)
        if old_resp_ids - new_resp_ids:
            to_remove = User.objects.filter(staff_id__in=old_resp_ids - new_resp_ids)
            user.remove_responsibles(to_remove)

    need_sync_indirect = False

    change_department = user_data['department_group']['id'] != getattr(user.department_group, 'external_id', None)
    # Обновим департаментную группу, если пользователь не уволен и поменял департаментную группу
    if change_department and not user_data['official']['is_dismissed']:
        need_sync_indirect = change_department_group(user, user_data['department_group']['id'])
    # При увольнении мы не сбрасываем департаментную группу,
    # если пользователь восстановился в том же департаменте - обновим его членство в группе
    elif not user.is_active and not user_data['official']['is_dismissed']:
        need_sync_indirect = user_hired_back(user)
    # если пользователя уволили - удалим его членство в группе
    elif user.is_active and user_data['official']['is_dismissed']:
        need_sync_indirect = user_dismissed(user)

    # updating user
    update_model(user, user_fields)
    return need_sync_indirect


@transaction.atomic
def user_hired_back(user):
    canonical_user = CanonicalMember(user.center_id, state=GROUPMEMBERSHIP_STATE.ACTIVE)
    user.department_group.add_members([canonical_user])
    return True


@transaction.atomic
def user_dismissed(user):
    canonical_user = CanonicalMember(user.center_id, state=GROUPMEMBERSHIP_STATE.ACTIVE)
    if user.department_group_id is None:
        raise ValueError(f"User's {user.username} department is None")
    group = Group.objects.prefetch_for_remove_members().filter(pk=user.department_group_id).first()
    group.remove_members([canonical_user])
    return True


@transaction.atomic
def change_department_group(user, new_group_external_id):
    source_group = user.department_group
    # ToDo: возможно стоит выше закешировать все группы, чтоб не ходить за ними в базу
    new_group = Group.objects.get_department_by_external_id(new_group_external_id)
    if new_group is None:
        log.error(
            'Error when move user %s to group with external_id %s, recieved nonexistent group',
            user.username,
            new_group_external_id,
        )
        return False
    user.department_group = new_group
    user.save(update_fields=['department_group'])

    log.info(
        '%s moved from %s to %s',
        user.username,
        'none' if source_group is None else source_group.slug,
        new_group.slug
    )

    user.actions.create(
        action='user_change_department',
        data={
            'department_from': None if source_group is None else source_group.id,
            'dep_name_from': 'Не указан' if source_group is None else source_group.name,
            'department_to': new_group.id,
            'dep_name_to': new_group.name,
        }
    )

    if source_group is not None:
        user.transfers.create(
            type='user',
            state='undecided',
            source=source_group,
            target=new_group,
            source_name=source_group.as_snapshot(),
            target_name=new_group.as_snapshot(),
        )

    canonical_user = CanonicalMember(user.center_id, state=GROUPMEMBERSHIP_STATE.ACTIVE)

    if source_group:
        group = Group.objects.prefetch_for_remove_members().get(pk=source_group.pk)
        group.remove_members([canonical_user])
    new_group.add_members([canonical_user])
    return True


def import_users(since=None):
    """Импортирует пользователей из Staff API.
    Если указан параметр since, то импортируются только сотрудники, чьи данные изменялись
    за последние N минут."""
    log.info('Importing users from Staff API')

    url = settings.STAFF_API_V3_URL.format(handle='persons')
    params = {'_fields': ','.join(STAFF_API_FIELDS), '_sort': 'id', '_limit': USER_FETCH_BATCH_SIZE}

    since_query = []
    if since:
        since = timezone.now() - timezone.timedelta(0, 60 * since)
        since = timezone.localtime(since, timezone.get_default_timezone())
        since_query = ['_meta.modified_at>"{}"'.format(since.strftime('%Y-%m-%dT%H:%M'))]

    start_id = 0
    users = []
    while True:
        id_query = ['id>{}'.format(start_id)]
        params['_query'] = ' and '.join(since_query + id_query)

        resp = http.get(url, params=params, headers={'Authorization': 'OAuth %s' % settings.IDM_STAFF_OAUTH_TOKEN})
        try:
            resp = resp.json()
        except ValueError:
            log.exception('Staff API returned incorrect response: %s', resp.content)
            raise RuntimeError('Staff API returned incorrect json response')

        users += resp['result']
        if resp['pages'] == 1 or not resp['result']:
            break
        start_id = resp['result'][-1]['id']

    user_cache = {
        user.username: user
        for user in User.objects.users().select_related('department_group').prefetch_related('responsibles')
    }

    def keyfunc(entry):
        user = user_cache.get(entry['login'])
        return user.pk if user else -1
    users.sort(key=keyfunc)

    need_sync_indirect = False
    for batch_index in range(0, len(users), USER_UPDATE_BATCH_SIZE):
        with transaction.atomic():
            for user in users[batch_index:batch_index + USER_UPDATE_BATCH_SIZE]:
                try:
                    need_sync_indirect |= update_user(user, user_cache)
                except Exception:
                    log.exception('Fail while updating %s', user['login'])

    if need_sync_indirect:
        sync_indirect_memberships(block=True)


def process_one_tvm_app(tvm_app_json, tvm_apps_cache, services_cache):
    fields = {
        'username': str(tvm_app_json['resource']['external_id']),
        'first_name': tvm_app_json['resource']['name'],
    }
    if fields['username'] in tvm_apps_cache:
        tvm_app = tvm_apps_cache[fields['username']]
        update_model(tvm_app, fields)
    else:
        tvm_app = User.objects.create(type=USER_TYPES.TVM_APP, **fields)
        tvm_apps_cache[fields['username']] = tvm_app

    fields = {
        'external_id': tvm_app_json['service']['id'],
        'slug': tvm_app_json['service']['slug'].lower(),
        'name': tvm_app_json['service']['name']['ru'],
        'name_en': tvm_app_json['service']['name']['en'],
    }
    if fields['external_id'] in services_cache:
        service = services_cache[fields['external_id']]
        update_model(service, fields)
    else:
        service = Group.objects.create(type=GROUP_TYPES.TVM_SERVICE, **fields)
        services_cache[fields['external_id']] = service

    membership_params = {
        'user': tvm_app,
        'group': service,
        'is_direct': True,
    }
    membership = GroupMembership.objects.filter(**membership_params).first()
    if not membership:
        GroupMembership.objects.create(state=GROUPMEMBERSHIP_STATE.ACTIVE, **membership_params)
    elif membership.state != GROUPMEMBERSHIP_STATE.ACTIVE:
        membership.state = GROUPMEMBERSHIP_STATE.ACTIVE
        membership.save(update_fields=['state'])

    return tvm_app.username


def import_tvm_apps():
    log.info('Importing tvm_apps from ABC API')
    tvm_apps_cache = {tvm_app.username: tvm_app for tvm_app in User.objects.tvm_apps()}
    services_cache = {service.external_id: service for service in Group.objects.filter(type=GROUP_TYPES.TVM_SERVICE)}
    processed_usernames = []
    url = '%s?%s' % (
        settings.ABC_RESOURCES_URL,
        urllib.parse.urlencode({
            'fields': 'resource.external_id,resource.name,service.id,service.slug,service.name',
            'type': settings.TVM_RESOURCE_TYPE,
            'state': 'granted',
            'page_size': 1000,
        })
    )

    while True:
        resp = http.get(url, headers={'Authorization': 'OAuth %s' % settings.IDM_STAFF_OAUTH_TOKEN})
        try:
            resp = resp.json()
        except ValueError:
            log.exception('ABC API returned incorrect response: %s', resp.content)
            raise RuntimeError('ABC API returned incorrect json response')

        for tvm_app_json in resp['results']:
            username = process_one_tvm_app(tvm_app_json, tvm_apps_cache, services_cache)
            processed_usernames.append(username)

        url = resp.get('next')
        if not url:
            break

    inactive_apps = User.objects.tvm_apps().exclude(username__in=processed_usernames)
    inactive_apps.update(is_active=False)
    GroupMembership.objects.filter(
        user__type=USER_TYPES.TVM_APP,
        user__in=inactive_apps,
        state=GROUPMEMBERSHIP_STATE.ACTIVE,
    ).update(state=GROUPMEMBERSHIP_STATE.INACTIVE)


def process_one_service(tvm_service):
    api_usernames = set()
    url = '%s?%s' % (
        settings.ABC_MEMBERS_URL,
        urllib.parse.urlencode({
            'service': tvm_service.external_id,
            'role__in': f'{settings.ABC_HEAD_ROLE},{settings.ABC_TVM_MANAGER_ROLE}',
            'fields': 'person.login',
            'page_size': 100,
        })
    )

    while True:
        resp = http.get(url, headers={'Authorization': 'OAuth %s' % settings.IDM_STAFF_OAUTH_TOKEN})
        try:
            resp = resp.json()
        except ValueError:
            log.exception('ABC API returned incorrect response: %s', resp.content)
            raise RuntimeError('ABC API returned incorrect json response')

        api_usernames.update(
            member['person']['login'] 
            for member in resp['results']
        )

        url = resp.get('next')
        if not url:
            break

    responsibilities = tvm_service.responsibilities.values_list('user__username', flat=True)
    old_active_usernames = set(responsibilities.active())
    old_inactive_usernames = set(responsibilities.inactive())

    tvm_service.responsibilities.filter(user__username__in=old_active_usernames - api_usernames).update(
        date_leaved=timezone.now(),
        is_active=False
    )

    new_usernames = api_usernames - old_inactive_usernames - old_active_usernames
    new_users = User.objects.filter(username__in=new_usernames)
    new_responsibilities = [
        GroupResponsibility(group=tvm_service, date_joined=timezone.now(), is_active=True, user=user)
        for user in new_users
    ]
    GroupResponsibility.objects.bulk_create(new_responsibilities)

    changed_usernames = api_usernames - new_usernames
    tvm_service.responsibilities.filter(
        user__username__in=changed_usernames,
    ).update(
        is_active=True,
        date_joined=timezone.now(),
    )


def import_tvm_responsibles():
    log.info('Importing tvm_apps responsibles from ABC API')
    for tvm_service in Group.objects.filter(type=GROUP_TYPES.TVM_SERVICE):
        try:
            process_one_service(tvm_service)
        except Exception:
            log.exception('Error in import_tvm_responsibles for service %s', tvm_service.slug)
