# coding: utf-8
"""
конфигурация части админки, относящейся к пользователям
"""


from ajax_select import make_ajax_form
from django.contrib import messages
from django.contrib.admin import site, ModelAdmin

from idm.framework.base_admin import IdmModelAdminMixin
from idm.sync.ldap.connector import NewLDAP
from idm.users.models import User


def ad_restore_user_action(modeladmin, request, queryset):
    """
    Восстановить уволенного в AD пользователя
    """
    if not request.user.has_perm('core.edit_users'):
        raise RuntimeError('only superuser can restore other users in AD')

    if queryset.model is not User:
        raise RuntimeError('can only work with User objects')

    with NewLDAP() as ldap:
        for user in queryset:
            try:
                ldap.restore_user(user.username, ad_reason_data={
                    'reason': 'из админки пользователем %s' % request.user.username
                })
                user.ldap_active = True
                user.ldap_blocked = False
                user.ldap_blocked_timestamp = None
                user.save()

            except RuntimeError as exc:
                modeladmin.message_user(request, str(exc), level=messages.ERROR)
            else:
                modeladmin.message_user(request, 'пользователь успешно восстановлен в AD')


ad_restore_user_action.short_description = 'Восстановить заблокированного в AD пользователя'


def ad_block_user_action(modeladmin, request, queryset):
    """
    Заблокировать пользователей в AD, оставив им группы и не перенося в Old Users
    """
    if not request.user.has_perm('core.edit_users'):
        raise RuntimeError('only superuser can block other users in AD')

    if queryset.model is not User:
        raise RuntimeError('can only work with User objects')

    with NewLDAP() as ldap:
        for user in queryset:
            try:
                ldap.disable_user(user, move_to_old=False, ad_reason_data={
                    'reason': 'из админки пользователем %s' % request.user.username
                })
                user.ldap_active = False
                user.ldap_blocked = True
                user.save()

            except RuntimeError as exc:
                modeladmin.message_user(request, str(exc), level=messages.ERROR)
            else:
                modeladmin.message_user(request, 'пользователь успешно заблокирован в AD')


ad_block_user_action.short_description = 'Заблокировать в AD пользователя'


def ad_disable_user_action(modeladmin, request, queryset):
    """
    Заблокировать пользователей в AD, убрать их из групп и перенести в Old Users
    """
    if not request.user.has_perm('core.edit_users'):
        raise RuntimeError('only superuser can dismiss other users in AD')

    if queryset.model is not User:
        raise RuntimeError('can only work with User objects')

    with NewLDAP() as ldap:
        for user in queryset:
            try:
                ldap.disable_user(user, move_to_old=True, ad_reason_data={
                    'reason': 'из админки пользователем %s' % request.user.username
                })
                user.ldap_active = False
                user.ldap_blocked = False
                user.save()

            except RuntimeError as exc:
                modeladmin.message_user(request, str(exc), level=messages.ERROR)
            else:
                modeladmin.message_user(request, 'пользователь успешно уволен в AD')


ad_disable_user_action.short_description = 'Уволить в AD пользователя'


class UserAdmin(IdmModelAdminMixin, ModelAdmin):
    """
    Пользователи IDM
    """
    editable = False
    list_display = (
        'username',
        'email',
        'first_name',
        'last_name',
        'is_staff'
    )
    search_fields = ('username',)
    exclude = ['user_permissions', 'password']
    actions = [ad_restore_user_action]


site.register(User, UserAdmin)
