# coding: utf-8


import logging
from collections import defaultdict
from typing import Set, Dict

from django.conf import settings
from django.contrib.auth.models import UserManager
from django.core.cache import cache
from django.db import models
from django.utils import timezone
from django.utils.translation import ugettext as _

from idm.closuretree.managers import CttManager
from idm.core.constants.email_templates import (
    PASSPORT_LOGIN_NEED_ATTACH_REGULAR_TEMPLATES,
    PASSPORT_LOGIN_NEED_ATTACH_NEW_MEMBER_TEMPLATES,
)
from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.exceptions import DismissedThresholdViolation
from idm.monitorings.metric import FiredUsersLimitExceededMetric
from idm.nodes.updatable import StatefulQuerySet
from idm.users.constants.group import GROUP_TYPES
from idm.users.constants.user import USER_TYPES

log = logging.getLogger(__name__)


class UserQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)

    def dismissed(self):
        return self.users().filter(is_active=False)

    def active_in_ad(self):
        """
        Вернуть QuerySet сотрудников, которые не уволены в AD.
        """
        return self.users().filter(ldap_active=True)

    def with_deprivable_roles(self):
        """
        Вернуть QuerySet сотрудников, у которых есть deprivable роли.
        То есть роли в RETURNABLE_STATES.
        """
        from idm.core.models import Role
        q_filter = Role.objects.deprivable_query(prefix='roles__')
        q_filter &= Role.objects.of_operational_system_query(prefix='roles__')
        qs = self.filter(q_filter).distinct()
        return qs

    def lock_by_select(self):
        return list(self.select_for_update().values_list('pk', flat=True))

    def users(self):
        return self.filter(type=USER_TYPES.USER)

    def tvm_apps(self):
        return self.filter(type=USER_TYPES.TVM_APP)


class BaseIDMUsersManager(models.Manager.from_queryset(UserQuerySet), UserManager):
    def get_idm_robot(self, fresh=False):
        robot = None
        if not fresh:
            robot = cache.get('robot_idm')
        if robot is None:
            robot = self.get(username=settings.IDM_ROBOT_USERNAME)
            robot._state.db = None
            cache.set('robot_idm', robot, timeout=settings.IDM_ROBOT_CACHE_TIMEOUT)
        return robot

    def block_dismissed_in_ldap(self, mode='fast', threshold=None, block=False):
        if mode == 'fast':
            self.block_dismissed_fast(threshold=threshold, block=block)
        else:
            # full is always synchronous
            self.block_dismissed_full(threshold=threshold)

    def check_dismissed_count(self, dismissed_count, threshold=None):
        if threshold is None:
            threshold = settings.IDM_DISMISSED_USERS_THRESHOLD
        if dismissed_count >= threshold:
            FiredUsersLimitExceededMetric.set(dismissed_count)
            log.info('Dismissed users threshold violation')
            raise DismissedThresholdViolation('Dismissed threshold violation')

        FiredUsersLimitExceededMetric.set(0)

    def block_dismissed_fast(self, threshold=None, block=False):
        if block:
            method_name = 'apply_async'
        else:
            method_name = 'apply'

        from idm.core.tasks import DisableAccount

        active_in_ad = self.dismissed().active_in_ad()
        dismissed_count = active_in_ad.count()
        self.check_dismissed_count(dismissed_count, threshold)

        active_in_ad = active_in_ad.values_list('username', flat=True)
        for username in active_in_ad:
            log.info('creating task to disable LDAP user %s account', username)
            task = DisableAccount
            method = getattr(task, method_name)
            method(kwargs={
                'username': username,
                'reason': _('Увольнение сотрудника по данным из staff'),
            })

    def block_dismissed_full(self, threshold=None):
        from idm.sync import newhire
        from idm.sync.ldap import connector
        from idm.notification.utils import report_problem

        log.info('Start blocking in AD already blocked users')
        reason = _('Повторная блокировка пользователя, которого уже блокировали ранее.')

        idm_users = self.dismissed()

        log.info('Get hired employees')
        try:
            hired_employees = newhire.get_hired_employees()
        except Exception:
            report_problem(
                _('Сверка уволенных пользователей не удалась из-за неответа Наниматора'),
                ['emails/service/incident/newhire_error.txt'],
                incident=True
            )
            return

        dismissed_but_active = set()
        has_effective_groups = set()
        with connector.NewLDAP() as ldap_connect:
            for user in idm_users:
                account = ldap_connect.search_user(user.username)

                if account is None:
                    log.info('User %s was not found in AD', user.username)
                    continue
                if account.has_effective_groups():
                    has_effective_groups.add(user)
                if account.is_fully_blocked():
                    continue
                if user.username in hired_employees:
                    log.info('User %s is inactive, but will be hired soon', user.username)
                    continue

                dismissed_but_active.add(user)

            dismissed_count = len(dismissed_but_active)
            self.check_dismissed_count(dismissed_count, threshold)

            failed_to_remove = set(has_effective_groups)
            failed_to_block = []
            for user in dismissed_but_active:
                log.info('User %s blocked in db, but active in AD. Trying to reblock.', user.username)
                removed_from_groups = None
                try:
                    deactivated, removed_from_groups, blocked = ldap_connect.disable_user(
                        user,
                        move_to_old=True,
                        ad_reason_data={'reason': reason}
                    )
                except Exception:
                    log.exception('Cannot block user %s in AD.' % user.username)
                    deactivated = None
                    blocked = None

                if removed_from_groups:
                    failed_to_remove -= {user}

                if deactivated:
                    user.ldap_active = False
                    user.ldap_blocked_timestamp = timezone.now()
                    user.save(update_fields=['ldap_active', 'ldap_blocked_timestamp'])
                    log.info('User %s was successfully blocked in AD.', user.username)
                    continue
                
                if not blocked:
                    failed_to_block.append(user)

                if deactivated is False:
                    log.error('Cannot block user %s in AD.' % user.username)

            for user in has_effective_groups - dismissed_but_active:
                if user.username in hired_employees:
                    failed_to_remove.remove(user)
                    continue
                account = ldap_connect.search_user(user.username)
                if account.remove_from_groups():
                    failed_to_remove.remove(user)

        if failed_to_remove:
            log.warning('Cannot remove users %s from AD groups.' % ', '.join(u.username for u in failed_to_remove))
            from idm.notification.utils import create_removing_from_ad_groups_issue
            issue = create_removing_from_ad_groups_issue(failed_to_remove)
            log.warning('Removing from groups failed. Issue %s created.' % issue.key)
        
        if failed_to_block:
            log.warning('Cannot block users %s in AD.' % ', '.join(u.username for u in failed_to_block))
            from idm.notification.utils import create_blocking_in_ad_issue
            issue = create_blocking_in_ad_issue(failed_to_block)
            log.warning('Blocking in AD failed. Issue %s created.' % issue.key)
            
        if bool(dismissed_but_active):
            log.info('Reblocking command found an incident')
        else:
            log.info('Reblocking command finished successfully')


class GroupQueryset(StatefulQuerySet):
    def nonroot(self):
        return self.filter(level__gt=0)

    def prefetch_for_remove_members(self):
        return self.prefetch_related('memberships__user')

    def prefetch_for_hashing(self):
        return self.select_related('parent').prefetch_related('memberships__user', 'responsibilities__user')

    def department(self):
        return self.filter(type='department')

    def user_groups(self):
        return self.filter(type__in=GROUP_TYPES.USER_GROUPS)

    def tvm_groups(self):
        return self.filter(type__in=GROUP_TYPES.TVM_GROUPS)

    def lock_by_select(self):
        return list(self.select_for_update().values_list('pk', flat=True))


class GroupManager(models.Manager.from_queryset(GroupQueryset), CttManager):
    def get_root(self, type_):
        return self.get_queryset().get(parent=None, type=type_)

    def get_scopes_of_groups(self, groups):
        return self.active().filter(parent__in=groups)

    def get_membership_query(self, users_qs):
        """Возвращает список групп, включая родительские, где хотя бы один из переданных пользователей – участник."""

        qs = self.active().filter(
            parent__isnull=False,
            groupclosure_children__child__memberships__user__in=users_qs,
            groupclosure_children__child__memberships__state__in=GROUPMEMBERSHIP_STATE.ACTIVE_STATES,
        )
        return qs

    def get_by_external_id(self, external_id, group_type=None):
        filter_query = models.Q(state__in=('active', 'depriving'), external_id=external_id)
        if group_type is not None:
            filter_query &= models.Q(type=group_type)
        qs = self.filter(filter_query)
        group = None
        try:
            group = qs.get()
        except (self.model.DoesNotExist, self.model.MultipleObjectsReturned):
            pass
        return group

    def get_department_by_external_id(self, external_id):
        return self.get_by_external_id(external_id, group_type=GROUP_TYPES.DEPARTMENT)


class ActiveMembershipManager(models.Manager):
    def get_queryset(self):
        return super(ActiveMembershipManager, self).get_queryset().filter(
            groupmembership__state=GROUPMEMBERSHIP_STATE.ACTIVE,
        )


class GroupResponsibilityQueryset(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)

    def inactive(self):
        return self.filter(is_active=False)


class GroupResponsibilityManager(models.Manager.from_queryset(GroupResponsibilityQueryset)):
    """Менеджер для объектов ответственности за группы"""


class GroupMembershipQuerySet(models.QuerySet):
    def direct(self):
        return self.filter(is_direct=True)

    def indirect(self):
        return self.filter(is_direct=False)

    def active(self):
        return self.filter(state__in=GROUPMEMBERSHIP_STATE.ACTIVE_STATES)

    def inactive(self):
        return self.filter(state__in=GROUPMEMBERSHIP_STATE.INACTIVE_STATES)


class GroupMembershipManager(models.Manager.from_queryset(GroupMembershipQuerySet)):
    def update_notified_about_passport_login(self, membership_ids):
        self.filter(pk__in=membership_ids).update(notified_about_passport_login=True)

    def send_passport_login_attach_reminders(
        self, templates,
        groups_id: Set[int] = None,
        users_per_groups: Dict[int, Set[int]] = None,
        exclude_notified=False,
    ):
        assert (groups_id is None) ^ (users_per_groups is None)

        if groups_id is None:
            groups_id = set(users_per_groups)

        if not groups_id:
            return

        memberships = (
            self
            .filter(passport_login__isnull=True, group_id__in=groups_id, state=GROUPMEMBERSHIP_STATE.ACTIVE)
            .select_related('group', 'user', 'user__department_group')
        )

        if exclude_notified:
            memberships = memberships.exclude(notified_about_passport_login=True)

        success_notified = []

        user_groups = defaultdict(list)
        for membership in memberships:
            if not users_per_groups or membership.user_id in users_per_groups[membership.group_id]:
                user_groups[membership.user].append(membership)

        for user, user_memberships in user_groups.items():
            result = user.send_email_about_attach_passport_login_to_membership(
                templates,
                {'groups_external_id': [membership.group.external_id for membership in user_memberships]},
            )
            if result:
                success_notified += [membership.pk for membership in user_memberships]
        self.update_notified_about_passport_login(success_notified)

    def send_regular_passport_login_attach_reminders(self):
        from idm.core.models import Role

        self.send_passport_login_attach_reminders(
            PASSPORT_LOGIN_NEED_ATTACH_REGULAR_TEMPLATES,
            groups_id=Role.objects.get_need_attach_passport_login_groups_id_in_aware_for_membership_systems()
        )

        self.send_passport_login_attach_reminders(
            PASSPORT_LOGIN_NEED_ATTACH_REGULAR_TEMPLATES,
            users_per_groups=Role.objects.get_need_attach_passport_login_groups_id_in_unaware_systems()
        )

    def send_passport_login_attach_reminders_to_new_members(self):
        from idm.core.models import Role

        groups_id = Role.objects.get_need_attach_passport_login_groups_id_in_aware_for_membership_systems()
        self.send_passport_login_attach_reminders(
            PASSPORT_LOGIN_NEED_ATTACH_NEW_MEMBER_TEMPLATES,
            groups_id=groups_id,
            exclude_notified=True,
        )

    def check_new_members_passport_logins(self):
        from idm.core.tasks import CheckGroupMembershipPassportLogin
        from idm.core.models import Role

        groups_id = Role.objects.get_need_attach_passport_login_groups_id_in_aware_for_membership_systems()
        memberships = (
            self
            .filter(passport_login__isnull=True, group_id__in=groups_id, state=GROUPMEMBERSHIP_STATE.ACTIVE)
            .exclude(notified_about_passport_login=True)
            .select_related('group', 'user')
        )

        for membership in memberships:
            CheckGroupMembershipPassportLogin.run(
                step='init',
                membership_id=membership.pk,
            )


class OrganizationQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)


class OrganizationManager(models.Manager.from_queryset(OrganizationQuerySet)):
    pass
