# coding: utf-8

from django.db import transaction

from idm.celery_app import app
from idm.core.models import Transfer
from idm.users.models import GroupMembership, GROUP_TYPES
from idm.framework.task import BaseTask
from idm.users.sync.groups import (
    sync_group_type,
    deprive_depriving_groups,
    sync_indirect_memberships,
)
from idm.utils.lock import lock, LockAlreadyAcquired


class SyncGroups(BaseTask):

    def init(self, block=False):
        with lock(self.task_name) as acquired:
            if not acquired:
                raise LockAlreadyAcquired

            for group_type in GROUP_TYPES.USER_GROUPS:
                with transaction.atomic():  # FIXME: ок ли?
                    sync_group_type(group_type=group_type, block=block)

            Transfer.objects.create_user_group_transfers()
            sync_indirect_memberships(block=True)


class DepriveDeprivingGroups(BaseTask):
    """
    Отзывает все членства и удаляет группы, которые исчезли
    со стаффа больше, чем IDM_DEPRIVING_GROUP_HOURS часов назад
    """
    def init(self):
        with lock(self.task_name) as acquired:
            if not acquired:
                raise LockAlreadyAcquired

            deprive_depriving_groups()


class SendLoginRemindersToNewGroupMembers(BaseTask):
    """
    К каждому членству в группе может быть привязан внешний паспортный логин, который отправляется
    вместе с раскрытой из групповой ролью, у ноды которой указано требование для отправки этого логина.
    Таска смотрит на новые членства в группах, для которых существует хотя бы одна роль с таким требованием
    и отправляет владельцам этих членств письма с просьбой привязать к группе паспортный логин.
    Работает только для ролей систем с политикой aware of memberships with logins (почему?).
    """
    def init(self):
        with lock(self.task_name) as acquired:
            if not acquired:
                raise LockAlreadyAcquired

            GroupMembership.objects.check_new_members_passport_logins()
            GroupMembership.objects.send_passport_login_attach_reminders_to_new_members()


SyncGroups = app.register_task(SyncGroups())
DepriveDeprivingGroups = app.register_task(DepriveDeprivingGroups())
SendLoginRemindersToNewGroupMembers = app.register_task(SendLoginRemindersToNewGroupMembers())
