from typing import Union
from django.conf import settings

from idm.utils.attributes import get_attribute
from idm.utils.i18n import get_lang_key, get_localized_fieldname
from idm.users.constants.group import GROUP_TYPES
from idm.users.constants.user import USER_TYPES


def get_group_url(group: Union['Group', dict]) -> Union[str, None]:
    group_type = get_attribute(group, ['type'])
    slug = get_attribute(group, ['slug'])
    context = {'slug': slug}

    if group_type == GROUP_TYPES.DEPARTMENT:
        template = settings.IDM_GROUP_LINK_TEMPLATE_DEPARTMENT

    elif group_type == GROUP_TYPES.WIKI:
        template = settings.IDM_GROUP_LINK_TEMPLATE_WIKI

    elif group_type == GROUP_TYPES.SERVICE:
        template = settings.IDM_GROUP_LINK_TEMPLATE_ABC
        level = None

        while True:
            # у родителя може быть не указан уровень, тогда считаем его сами
            level = get_attribute(group, ['level']) or (level - 1 if level else None)
            if not level:
                return None
            if level == 1:
                break
            group = get_attribute(group, ['parent'])

        slug = get_attribute(group, ['slug'])
        if not slug:
            return None
        if slug.startswith('svc_'):
            slug = slug.replace('svc_', '', 1)
        context = {'slug': slug}

    else:
        raise ValueError(f'Wrong group type `{group_type}`')

    return template % context


def get_localized_group_name(group, lang=None):
    field = get_localized_fieldname('name', lang=lang)
    return get_attribute(group, [field])


def get_user_fullname(user, lang=None):
    if get_attribute(user, ['type']) == USER_TYPES.TVM_APP:
        return get_attribute(user, ['first_name'])

    if lang is None:
        lang = get_lang_key()

    first_name = get_attribute(user, ['first_name'])
    last_name = get_attribute(user, ['last_name'])

    if first_name and last_name:
        if lang == 'ru':
            return '{0} {1}'.format(first_name, last_name).strip()
        else:
            return '{0} {1}'.format(get_attribute(user, ['first_name_en']), get_attribute(user, ['last_name_en'])).strip()
    else:
        return get_attribute(user, ['username'])
