# coding: utf-8
from __future__ import unicode_literals

import copy
import sys
import traceback
from contextlib import contextmanager
from types import TracebackType
from typing import Type, Optional, Dict, Any, Union, Callable

from idm.core.constants.action import ACTION_DATA_KEYS, ACTION_STATE
from idm.core.constants.errors import ERROR_LEN
from idm.core.models import Action


@contextmanager
def parent_action_completion(parent_action):
    if parent_action is None:
        yield
        return
    data = parent_action.data or {}
    data[ACTION_DATA_KEYS.ACTION_STATE] = ACTION_STATE.BEGIN
    parent_action.data = data
    parent_action.save(update_fields=['data'])
    try:
        yield
        parent_action.data[ACTION_DATA_KEYS.ACTION_STATE] = ACTION_STATE.COMPLETED
    except Exception:
        parent_action.data[ACTION_DATA_KEYS.ACTION_STATE] = ACTION_STATE.FAILED
    finally:
        parent_action.save(update_fields=['data'])


def format_error_traceback(
        exc_type: Type[Exception] = None,
        exc_value: Exception = None,
        tb: TracebackType = None,
        max_length: Optional[int] = ERROR_LEN,
) -> str:
    if not all((exc_type, exc_value, tb)):
        exc_type, exc_value, tb = sys.exc_info()

    tb_lines = traceback.format_exception(exc_type, exc_value, tb)
    length = sum(len(x) for x in tb_lines)
    pop_idx = 3  # для сокращения выдираем строчки из начала (но оставляем первые три)
    if max_length:
        if len(tb_lines) <= 3:
            return ''.join(tb_lines)[-max_length:]
        shorted = False
        while length + 4 > max_length:
            shorted |= True
            line = tb_lines.pop(pop_idx)
            length -= len(line)
        if shorted:
            tb_lines.insert(pop_idx, '...\n')

    return ''.join(tb_lines)


class ActionWrapManager:
    def __init__(
            self,
            start_action: str,
            finish_action: str,
            *,
            fail_action: str = None,
            extra: Dict[str, Any] = None,

    ):
        self._start_key = start_action
        self._finish_key = finish_action
        self._fail_key = fail_action or finish_action
        self._extra = extra and copy.deepcopy(extra) or {}
        self.start_action = None
        self.finish_action = None

        self.success_data = {}
        self.failure_data = {}
        self.success_callbacks = []
        self.failure_callbacks = []

    def __enter__(self):
        from idm.core.models import Action
        action = Action.objects.create(action=self._start_key, **self._extra)
        self.start_action = action
        return self

    def on_success(self, data: Union[Callable, Dict[str, Any]]):
        if callable(data):
            self.success_callbacks.append(data)
        else:
            self.success_data.update(data)

    def on_failure(self, data: Union[Callable, Dict[str, Any]]):
        if callable(data):
            self.failure_callbacks.append(data)
        else:
            self.failure_data.update(copy.deepcopy(data))

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_val is None:
            action = self._finish_key
            data = {**copy.deepcopy(self.success_data), **self._extra}
            callbacks = self.success_callbacks
        else:
            action = self._fail_key
            action_data = self.failure_data.pop('data', {})
            action_data['traceback'] = format_error_traceback(exc_type, exc_val, exc_tb, max_length=None)
            data = {**copy.deepcopy(self.failure_data), **self._extra, 'data': action_data}
            callbacks = self.failure_callbacks

        for callback in callbacks:
            callback(self)
        self.finish_action = Action.objects.create(action=action, **data)
        return True


start_stop_actions = ActionWrapManager
