# coding: utf-8


import logging
import random
import time

from django.conf import settings
from redis.exceptions import ConnectionError, TimeoutError, ReadOnlyError
from redis.sentinel import Sentinel, MasterNotFoundError
from redis_cache.backends.base import parse_connection_kwargs, BaseRedisCache
from redis_cache.backends.single import RedisCache

log = logging.getLogger(__name__)

SENTINEL_PORT = 26379


class RedisCacheErrorInterceptor(object):

    def __init__(self, method_name):
        self.method_name = method_name

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.exception = exc_val
        if isinstance(self.exception, ConnectionError):
            log.warning('Could not connect to Redis while doing cache.{}'.format(self.method_name))
        elif isinstance(self.exception, TimeoutError):
            log.warning('Timeout from Redis while doing cache.{}'.format(self.method_name))
        elif isinstance(self.exception, ReadOnlyError):
            log.warning('Cannot write against a read only replica.')
        elif isinstance(self.exception, IndexError) or isinstance(self.exception, AttributeError):
            log.warning('Redis client is undefined')
        else:
            return False

        return True


def get_master_host(hosts):
    hosts = [(host.split(':')[0], SENTINEL_PORT) for host in hosts]
    sentinel = Sentinel(hosts, socket_timeout=settings.REDIS_TIMEOUT)
    try:
        host, port = sentinel.discover_master(settings.REDIS_DB)
    except MasterNotFoundError:
        return None
    else:
        return '{}:{}'.format(host, port)


def get_slave_hosts(hosts):
    hosts = [(host.split(':')[0], SENTINEL_PORT) for host in hosts]
    sentinel = Sentinel(hosts, socket_timeout=settings.REDIS_TIMEOUT)
    servers = sentinel.discover_slaves(settings.REDIS_DB)
    prepared_servers = [
        '{}:{}'.format(host, port)
        for host, port
        in servers
    ]
    return prepared_servers


class AutodiscoveringRedisCache(RedisCache):
    def __init__(self, server, params):
        BaseRedisCache.__init__(self, server, params)
        self.raw_servers = self.servers
        self.refresh_clients()

    def should_refresh_clients(self):
        return time.time() - self.last_config_refresh > settings.REDIS_CLIENTS_CONFIG_TTL

    def refresh_clients(self):
        master = get_master_host(self.raw_servers)
        # Динамически получаем список доступных хостов
        self.servers = ([master] if master else []) + get_slave_hosts(self.raw_servers)
        self.clients = {}
        for server in self.servers:
            client = self.create_client(server)
            self.clients[client.connection_pool.connection_identifier] = client
        self.client_list = list(self.clients.values())
        # Сохраняем мастер
        if master:
            kwargs = parse_connection_kwargs(master, db=self.db)
            self._master_client = self.clients[(
                kwargs['host'],
                kwargs['port'],
                kwargs['db'],
                kwargs['unix_socket_path'],
            )]
        else:
            self._master_client = None
            log.warning('Could not determine Redis master')
        # Обновляем timestamp
        self.last_config_refresh = time.time()

    @property
    def master_client(self):
        if self.should_refresh_clients():
            self.refresh_clients()
        return self._master_client

    def get_client(self, key, write=False):
        if self.should_refresh_clients():
            self.refresh_clients()
        if write:
            client = self.master_client
            if client is not None:
                return client
        return random.choice(list(self.client_list))

    def get(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('get') as interceptor:
            result = super(AutodiscoveringRedisCache, self).get(*args, **kwargs)
        if interceptor.exception:
            return None
        else:
            return result

    def set(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('set') as interceptor:
            super(AutodiscoveringRedisCache, self).set(*args, **kwargs)
        if interceptor.exception:
            return False
        else:
            return True

    def add(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('add') as interceptor:
            result = super(AutodiscoveringRedisCache, self).add(*args, **kwargs)
        if interceptor.exception:
            return False
        else:
            return result

    def delete(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('delete') as interceptor:
            result = super(AutodiscoveringRedisCache, self).delete(*args, **kwargs)
        if interceptor.exception:
            return False
        else:
            return result

    def get_many(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('get_many') as interceptor:
            result = super(AutodiscoveringRedisCache, self).get_many(*args, **kwargs)
        if interceptor.exception:
            return {}
        else:
            return result

    def set_many(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('set_many'):
            super(AutodiscoveringRedisCache, self).set_many(*args, **kwargs)

    def delete_many(self, *args, **kwargs):
        with RedisCacheErrorInterceptor('delete_many'):
            super(AutodiscoveringRedisCache, self).delete_many(*args, **kwargs)
