# coding: utf-8
import logging
from typing import Dict, Any, Union

import yarl
from django.conf import settings
import requests
from requests import RequestException  # NOQA
from requests.adapters import HTTPAdapter

from idm.utils import tvm

log = logging.getLogger(__name__)


def validate_url(url: str) -> yarl.URL:
    if url is None:
        raise ValueError('Url not set')

    url = yarl.URL(url)
    if url.scheme is None:
        raise ValueError('Url scheme not set')
    elif url.scheme not in ('http', 'https'):
        raise ValueError('Url scheme is not http-like')
    elif url.host is None:
        raise ValueError('Url host not set')

    return url


def _perform(
        url: Union[str, yarl.URL],
        method: str = 'GET',
        params: Dict[str, Any] = None,
        data: Dict[str, Any] = None,
        json: Union[dict, list, str] = None,
        use_client_certificate: bool = True,
        tvm_id: str = None,
        oauth_token: str = None,
        check_server_certificate: bool = True,
        timeout: float = settings.DEFAULT_REQUEST_TIMEOUT,
        headers: Dict[str, str] = None,
        max_retries: int= None,
        allow_redirects: bool = False,
) -> requests.Response:
    """
    Выполнить http(s)-запрос.

    @param url: URL
    @param method: Метод
    @param params: get-параметры
    @param data: POST-данные (dict или строка)
    @param use_client_certificate: Использовать клиентский сертификат
    @param check_server_certificate: Проверять серверный сертификат
    @param timeout: Таймаут
    @param headers: dict с заголовками. Будет дополнено дефолтными значениями `DEFAULT_HEADERS`.
        Если значение пустое, заголовок не будет использоваться, даже если есть такой дефолтный.
    @return: Response() с данными
    @raises: ResponseException
    """
    log.debug('Attempt to perform a request by requests: url=%s, method=%s, data=%s, use_client_certificate=%s, '
              'timeout=%s',
              repr(url), repr(method), repr(data), repr(use_client_certificate),
              repr(timeout))

    # Заголовки
    final_headers = settings.DEFAULT_REQUEST_HEADERS.copy()
    final_headers.update(headers or {})

    request_params = {
        'params': params,
        'headers': final_headers,
        'timeout': timeout,
        'verify': False,  # TODO: проверить и выставить в True
        'allow_redirects': allow_redirects,
    }

    # Определяем тело запроса
    if data is not None and json is not None:
        raise AssertionError('data and json parameters should not be defined at the same time')
    if json is not None:
        request_params['headers']['content-type'] = 'application/json'
        request_params['json'] = json
    else:
        request_params['data'] = data

    # Проверка серверного сертификата
    if check_server_certificate:
        request_params['verify'] = settings.CA_BUNDLE

    # Клиентский сертификат
    if tvm_id:
        request_params['headers']['X-Ya-Service-Ticket'] = tvm.get_tvm_ticket(tvm_id)
    elif oauth_token:
        request_params['headers']['Authorization'] = f'OAuth {oauth_token}'
    elif use_client_certificate:
        request_params['cert'] = settings.CLIENT_CERT

    try:
        with requests.sessions.Session() as session:
            if max_retries is not None:
                adapter = HTTPAdapter(max_retries=max_retries)
                session.mount('https://', adapter)
            response = session.request(method=method, url=str(url), **request_params)
    except Exception:
        log.warning('Failed to perform a request to "%s"', url, exc_info=True)
        raise

    log.debug('Received response with code "%s"', response.status_code)
    return response


def head(url: Union[str, yarl.URL], **kwargs):
    return _perform(url, 'HEAD', **kwargs)


def get(url: Union[str, yarl.URL], **kwargs):
    return _perform(url, 'GET', **kwargs)


def delete(url: Union[str, yarl.URL], **kwargs):
    return _perform(url, 'DELETE', **kwargs)


def post(url: Union[str, yarl.URL], data: Dict[str, Any] = None, json: Union[list, dict] = None, **kwargs):
    return _perform(url, 'POST', data=data, json=json, **kwargs)