# coding: utf-8
# Утилиты преобразования названия ролей на человеческий язык

import hashlib
import logging
from typing import Union

from django.conf import settings
from django.core.cache import cache
from django.utils.encoding import force_text, force_bytes
from django.utils.translation import ugettext as _

from idm.users.models import User, Group
from idm.utils import http

log = logging.getLogger(__name__)


_inflection_forms = {
    'кто': 'им',
    'что': 'им',
    'кого': 'род',
    'чего': 'род',
    'кому': 'дат',
    'чему': 'дат',
    'кого-что': 'вин',
    'кем': 'твор',
    'чем': 'твор',
    'о ком': 'пр',
    'о чем': 'пр',
    'где': 'пр',
}


def inflect(form: str, word: str, timeout: int = 3, fio: bool = False) -> str:
    """
    Вернуть словоформу согласно указанному падежу. Работает с помощью сервиса reqwizard.
    https://wiki.yandex-team.ru/poiskovajaplatforma/lingvistika/wizard/wizardasservis/inflector

    :type form: unicode
    :param form: вспомогательное слово падежа согласно таблице _inflection_forms
    :type word: unicode
    :param word: слово, форму которого надо вернуть
    :type timeout: int
    :param timeout: таймаут на http запрос к reqwizard
    :type fio: boolean
    :param fio: является ли переданное слово именем-фамилией

    :rtype: unicode
    """
    return _inflect(form, word, timeout, fio)


def _inflect(form: str, word: str, timeout: int = 3, fio: bool = False) -> str:
    """
    Эта функция нужна, чтобы ещё мокать в тестах
    """
    text = force_text(word)
    # именительный всегда совпадает с самим словом, экономим походы в кеш/сеть
    if form == 'кто':
        return word

    try:
        cache_key = hashlib.md5(force_bytes(text)).hexdigest()
        word_forms = cache.get(cache_key)
        word_forms = word_forms or {}

        if fio:
            text += ';fio=1'

        inflector_params = {
            'action': 'inflect',
            'text': text,
            'wizclient': 'idm',
        }
        url = settings.INFLECT_URL

        if not word_forms:
            resp_data = http.get(url, params=inflector_params, timeout=timeout, use_client_certificate=False).json()

            if 'Form' not in resp_data:
                # inflector doesn't know any word forms
                log.warning('inflector returned incorrect response: %s for %s', repr(resp_data), repr(text))
            else:
                for form_dct in resp_data['Form']:
                    word_forms[form_dct['Grammar'].lower()] = form_dct['Text']

            # write to the cache for 1 month
            cache.set(cache_key, word_forms, timeout=settings.INFLECT_TTL)

        if not word_forms:
            return word
        else:
            return force_text(word_forms[_inflection_forms[form]])

    except Exception:
        log.warning('Error while using inflector, returning uninflected value: %s', word, exc_info=1)
        return word


def format_username(subject_or_login: Union[str, int, User, Group], form: str) -> str:
    from idm.core.workflow.common.subject import Subject

    if subject_or_login is None:
        return _('Робот')

    try:
        if isinstance(subject_or_login, Subject):
            subject = subject_or_login.object
        elif isinstance(subject_or_login, str):
            subject = User.objects.get(username=subject_or_login)
        elif isinstance(subject_or_login, int):
            subject = Group.objects.active().get(external_id=subject_or_login)
        else:
            subject = subject_or_login

        ident_format = '({})'
        display_name = ''
        if isinstance(subject, User):
            display_name = inflect(form, subject.get_full_name(), fio=True)
        elif isinstance(subject, Group):
            display_name = inflect(form, subject.get_name())

        ident = get_ident(subject)
        if display_name and ident:
            if display_name == ident:  # get_full_name возвращает username
                return display_name
            return ' '.join((display_name, ident_format.format(ident)))
        elif not display_name and ident:
            return ident
        elif not ident:
            log.warning('Subject %s:%s with no identity', type(subject).__name__, subject)
            return display_name

        return subject_or_login
    except (User.DoesNotExist, Group.DoesNotExist):
        return subject_or_login


def get_ident(subject: Union[User, Group]) -> Union[str, int]:
    return (subject.get_ident()
            if getattr(subject, 'get_ident', None) is not None
            else getattr(subject, 'username', None) or getattr(subject, 'external_id', None))
