# coding: utf-8


from contextlib import contextmanager, ContextDecorator

import waffle
from django.conf import settings
from django_replicated.dbchecker import db_is_alive
from django_replicated.utils import routers


def is_in_manual_read_only():
    """Сервис находится в ручном read-only режиме"""

    return waffle.switch_is_active('idm.readonly')


def databases_for_reading_left():
    """
    Вернуть False, если не осталось базы из которой можно читать.

    Для сервиса наличие базы на чтение критично.

    @rtype: bool
    """

    db_instance_name = routers.db_for_read()
    if db_instance_name == routers.DEFAULT_DB_ALIAS:
        # django_replicated не проверяет мастер на живость в db_for_read
        if not db_is_alive(db_instance_name, number_of_tries=3):
            # мы попытаемся сходить в мертвый мастер. Баз на чтение не осталось
            return False
    return True


@contextmanager
def use_master():
    try:
        routers.use_state('master')
        yield
    finally:
        routers.revert()


class use_slave(ContextDecorator):
    def __enter__(self):
        if settings.USE_DB_REPLICAS:
            routers.use_state('slave')
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if settings.USE_DB_REPLICAS:
            routers.revert()
        return False
