# coding: utf-8
import contextlib
import itertools
import re
import sys
from typing import Union

from django.contrib.postgres.search import SearchQuery
from django.db import connection
from django.db.models import Func, Value, Expression
from django.db.utils import IntegrityError
from django.utils import six

from idm.core.workflow.exceptions import RoleAlreadyExistsError


def get_sql_false():
    if connection.vendor == 'postgresql':
        return 'FALSE'
    else:
        return '0'


def get_sql_true():
    if connection.vendor == 'postgresql':
        return 'TRUE'
    else:
        return '1'


@contextlib.contextmanager
def wrap_unique_violation(role):
    if connection.vendor == 'postgresql':
        prefix = 'duplicate key value violates unique constraint "idm_role_unique_'
    else:
        # sqlite
        prefix = 'UNIQUE constraint failed'
    try:
        yield
    except IntegrityError as e:
        if str(e).startswith(prefix):
            six.reraise(RoleAlreadyExistsError, RoleAlreadyExistsError(role), sys.exc_info()[2])
        else:
            raise


def generate_case(parent, user, fields_data, concurrently):
    null_for_indexname = {
        True: 'notnull',
        False: 'null'
    }
    null_for_condition = {
        True: 'IS NOT NULL',
        False: 'IS NULL',
    }

    index_name_template = 'idm_role_unique_parent_{parent}_user_{user}_fields_data_{fields_data}'
    index_name = index_name_template.format(
        parent=null_for_indexname[parent],
        user=null_for_indexname[user],
        fields_data=null_for_indexname[fields_data],
    )

    conditions = {
        'parent_id': null_for_condition[parent],
        'user_id': null_for_condition[user],
        'group_id': null_for_condition[not user],
        'fields_data': null_for_condition[fields_data]
    }

    field_names = ['node_id']

    if parent:
        field_names.append('parent_id')
    if user:
        field_names.append('user_id')
    else:
        field_names.append('group_id')
    if fields_data:
        field_names.append('fields_data')

    forward_sql = '''
        CREATE UNIQUE INDEX {maybe_concurrently} IF NOT EXISTS {index_name}
        ON upravlyator_role ({field_names})
        WHERE (
            is_returnable = {true} AND
            parent_id {conditions[parent_id]} AND
            user_id {conditions[user_id]} AND
            group_id {conditions[group_id]} AND
            fields_data {conditions[fields_data]}
        );
    '''.format(
        index_name=index_name,
        field_names=','.join(field_names),
        maybe_concurrently='CONCURRENTLY' if concurrently else '',
        true=get_sql_true(),
        conditions=conditions,
    )

    backward_sql = 'DROP INDEX {maybe_concurrently} IF EXISTS {index_name}'.format(
        maybe_concurrently='CONCURRENTLY' if concurrently else '',
        index_name=index_name,
    )

    return forward_sql, backward_sql


def generate_index_for_roles(concurrently):
    for parent, user, fields_data in itertools.product([True, False], [True, False], [True, False]):
        forwards, backwards = generate_case(
            parent=parent,
            user=user,
            fields_data=fields_data,
            concurrently=concurrently
        )
        yield forwards, backwards


def generate_index_for_nodes():
    forwards = """
    CREATE UNIQUE INDEX idm_rolenode_slugpath_uniq_where
    ON upravlyator_rolenode (system_id, slug_path) WHERE state='active';
    """
    backwards = """
    DROP INDEX idm_rolenode_slugpath_uniq_where;
    """
    return forwards, backwards


def generate_indexes_for_node_responsibility():
    forwards = """
    CREATE UNIQUE INDEX upravlyator_noderesponsibility_uniq
    ON upravlyator_noderesponsibility (node_id, user_id) WHERE is_active;
    """
    backwards = """
    DROP INDEX upravlyator_noderesponsibility_uniq;
    """
    return forwards, backwards


def generate_index_for_rolealias():
    forwards = """
    CREATE UNIQUE INDEX IF NOT EXISTS upravlyator_rolealias_uniq
    ON upravlyator_rolealias (node_id, type, name, name_en) WHERE is_active;
    """
    backwards = """
    DROP INDEX IF EXISTS upravlyator_rolealias_uniq;
    """
    return forwards, backwards


class RegexReplaceExpression(Func):
    function = 'regexp_replace'

    def __init__(self, value: Union[str, Expression], pattern: str, replace_to: str, *flags: str, **extra):
        if flags:
            flags = [Value(''.join(flags))]
        super().__init__(value, Value(pattern), Value(replace_to), *flags, **extra)


class PrefixSearchQuery(SearchQuery):
    """
    Оборачивает поисковый запрос с заданной конфигурацией в to_tsquery.
    Запрос разбивается по пробельным символам на элементы, по которым осуществляется конъюнктивный префиксный поиск.
    """
    config: Func
    invert: bool

    SANITIZE_REGEXP = r'[^A-Za-z0-9]'

    def as_sql(self, compiler, *_, **__):
        params = [self.value]
        if self.config:
            config_sql, params = compiler.compile(self.config)  # type: str, list
            template = 'to_tsquery({}::regconfig, %s)'.format(config_sql)

            prefixes = []
            for prefix in re.sub(self.SANITIZE_REGEXP, ' ', self.value).split():
                prefixes.append(f'{prefix}:*')
            params.append(' & '.join(prefixes))
        else:
            template = 'plainto_tsquery(%s)'
        if self.invert:
            template = '!!({})'.format(template)
        return template, params
