# coding: utf-8
"""
WSGI config for idm project.

It exposes the WSGI callable as a module-level variable named ``application``.

For more information on this file, see
https://docs.djangoproject.com/en/1.8/howto/deployment/wsgi/
"""
import uuid

import django
from django.conf import settings
from django.db import connections
from django_pgaas.wsgi import IdempotentWSGIHandler
from django_tools_log_context import request_log_context

from idm.manage import _setup_environ
from idm.utils import reverse
from idm.utils.log import no_log_context

_setup_environ()

# Close connections on WSGI fork
try:
    from uwsgidecorators import postfork
except ImportError:
    pass
else:
    postfork(connections.close_all)


def get_uwsgi_worker_id():
    try:
        import uwsgi
        # При запуске не через UWSGI модуль не попадает в контекст выполнения
        return uwsgi.worker_id()
    except ImportError:
        pass


class IdmWSGIHandler(IdempotentWSGIHandler):
    def get_response(self, request):
        batch_url = reverse('api_dispatch_list', api_name='v1', resource_name='batch')

        if request.path == batch_url and settings.IDM_BATCH_REQUEST_HEADER not in request.META:
            request.META[settings.IDM_BATCH_REQUEST_HEADER] = uuid.uuid1().hex

        log_context_manager = request_log_context
        if request.path == '/ping':
            # не пишем кучу логов на /ping
            log_context_manager = no_log_context

        with log_context_manager(request) as log_context:
            response = super().get_response(request)

            request_context = log_context.context.get('request', {})
            request_context['uwsgi_worker'] = get_uwsgi_worker_id()
            request_context['status_code'] = response.status_code
            if hasattr(response, '_log_additional_fields'):
                request_context['_additional_fields'] = response._log_additional_fields
                delattr(response, '_log_additional_fields')
            return response


def get_wsgi_application():
    django.setup()
    handler = IdmWSGIHandler()
    return handler


application = get_wsgi_application()
