package ru.yandex.intranet.imscore.core.services.identity.impl

import liquibase.pro.packaged.it
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Transactional
import ru.yandex.intranet.imscore.core.domain.identity.Identity
import ru.yandex.intranet.imscore.core.domain.identity.IdentityData
import ru.yandex.intranet.imscore.core.domain.identity.IdentityIdOneOf
import ru.yandex.intranet.imscore.core.domain.identity.specification.CreateIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.DeleteIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.GetIdentityByExternalIdSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.GetIdentityByIdSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentityChildListSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentityListSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.ModifiableIdentityDataSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.MoveIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.UpdateIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.IdentityRelation
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.InnerListIdentityGroupsSpecification
import ru.yandex.intranet.imscore.core.domain.identityType.IdentityType
import ru.yandex.intranet.imscore.core.exceptions.identity.IdentityAlreadyExistsException
import ru.yandex.intranet.imscore.core.exceptions.identity.IdentityIsNotGroupException
import ru.yandex.intranet.imscore.core.ports.identity.IdentityRepository
import ru.yandex.intranet.imscore.core.ports.identityRelation.IdentityRelationRepository
import ru.yandex.intranet.imscore.core.ports.identityType.IdentityTypeRepository
import ru.yandex.intranet.imscore.core.services.identity.IdentityService
import ru.yandex.intranet.imscore.core.util.common.CommonUtils.INNER_PAGING_SIZE
import ru.yandex.intranet.imscore.core.util.common.CommonUtils.getAllPages
import java.time.Instant
import java.util.Optional
import java.util.UUID

/**
 * Identity service implementation
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Service
open class IdentityServiceImpl(
    private val identityRepository: IdentityRepository,
    private val identityTypeRepository: IdentityTypeRepository,
    private val identityRelationRepository: IdentityRelationRepository,
): IdentityService {

    @Transactional(readOnly=true)
    override fun getListBySpec(spec: IdentityListSpecification): List<Identity> {
        val identityList = if (spec.hasGroupId()) {
            val group = identityRepository.getById(GetIdentityByIdSpecification(spec.groupId, false))

            if (!group.type.isGroup) {
                throw IllegalArgumentException(String.format("%s is not a group", group.id.toString()))
            }

            identityRepository.findAllIdentitiesByGroupId(spec)
        } else {
            identityRepository.getListBySpec(spec)
        }
        return identityList
    }

    @Transactional(readOnly=true)
    override fun getById(spec: IdentitySpecification): Identity {
        val idOneOf = spec.idOneOf
        if (idOneOf.hasId()) {
            return identityRepository.getById(
                GetIdentityByIdSpecification(
                    idOneOf.id.get(), spec.withData
                )
            )
        }
        val externalId = idOneOf.externalId.get()
        return identityRepository.getByExternalIdAndTypeId(
            GetIdentityByExternalIdSpecification(
                externalId.externalId, externalId.typeId, spec.withData
            )
        )
    }

    @Transactional(readOnly=false)
    override fun create(spec: CreateIdentitySpecification): Identity {
        val parentTypePair = validateCreateRequest(spec)
        val parentIdentity = parentTypePair.first

        val externalId = spec.identityExternalId
        val identity = Identity(
            externalId = externalId.externalId,
            type = parentTypePair.second,
            parentId = parentIdentity?.id,
            data = toIdentityData(spec.identityDataSpecification),
        )
        val createdIdentity = identityRepository.save(identity)
        if (parentIdentity != null) {
            val groupId = parentIdentity.id!!
            identityRelationRepository.save(
                IdentityRelation(
                    createdIdentity.id!!,
                    groupId,
                    IdentityRelation.ConnectionType.OWNERSHIP
                )
            )

            val groupPages = getAllGroups(groupId)
            val ids = mutableListOf(groupId)
            groupPages.forEach {
                ids.add(it.groupId)
            }

            identityRepository.updateIdentityModifiedAt(ids, createdIdentity.modifiedAt!!)

        }
        return createdIdentity
    }

    @Transactional(readOnly=false)
    override fun update(spec: UpdateIdentitySpecification): Identity {
        val identityIdOneOf = spec.identityIdOneOf
        val oldIdentity = if (identityIdOneOf.hasId()) {
            identityRepository.getByIdForUpdate(
                identityIdOneOf.id.get()
            )
        } else {
            val externalId = identityIdOneOf.externalId.get()
            identityRepository.getByExternalIdAndTypeIdForUpdate(
                externalId.externalId,
                externalId.typeId
            )
        }

        val oldIdentityWithData = identityRepository.getById(GetIdentityByIdSpecification(oldIdentity.id, true))

        val updatedIdentity = Identity(
            id = oldIdentityWithData.id, externalId = oldIdentityWithData.externalId, type = oldIdentityWithData.type,
            createdAt = oldIdentityWithData.createdAt, modifiedAt = oldIdentityWithData.modifiedAt,
            data = mergeIdentityData(oldIdentityWithData.data, spec.identityDataSpecification)
        )

        return identityRepository.save(updatedIdentity)
    }

    @Transactional(readOnly=false)
    override fun delete(spec: DeleteIdentitySpecification) {
        val groupIds = mutableSetOf<UUID>()
        val idOneOf = spec.identityIdOneOf

        val identity = if (idOneOf.hasId()) {
            identityRepository.getById(GetIdentityByIdSpecification(
                idOneOf.id.get(), false
            ))
        } else {
            val externalId = idOneOf.externalId.get()
            identityRepository.getByExternalIdAndTypeId(
                GetIdentityByExternalIdSpecification(externalId.externalId, externalId.typeId, false)
            )
        }

        val id = identity.id!!
        groupIds.addAll(getAllGroups(id)
            .map { it.groupId })

        if (identity.type.isGroup) {
            addAllChildrenGroups(id, groupIds)
        }

        identityRepository.deleteById(id)

        if (groupIds.isNotEmpty()) {
            identityRepository.updateIdentityModifiedAt(groupIds.toList(), Instant.now())
        }
    }

    @Transactional(readOnly=false)
    override fun move(spec: MoveIdentitySpecification) {
        val identity = getIdentityByIdSpecification(spec.identity)
        val group = getIdentityByIdSpecification(spec.toGroup)

        if (!group.type.isGroup) {
            throw IdentityIsNotGroupException(group.id!!)
        }
        if (group.id == identity.parentId) {
            return
        }

        val updatedGroupIds = mutableListOf<UUID>()

        if (identity.parentId != null) {
            val currentParent = getIdentityByIdSpecification(IdentityIdOneOf(
                identity.parentId
            ))
            identityRelationRepository.deleteByIdentityIdInAndGroupIdAndConnectionType(
                listOf(identity.id!!), currentParent.id!!, IdentityRelation.ConnectionType.OWNERSHIP
            )

            updatedGroupIds.add(identity.parentId!!)
            updatedGroupIds.addAll(getAllGroups(identity.parentId!!)
                .map { it.groupId })
        }
        identity.parentId = group.id
        identityRepository.save(identity)
        identityRelationRepository.upsert(
            IdentityRelation(
                identity.id!!,
                group.id!!,
                IdentityRelation.ConnectionType.OWNERSHIP
            )
        )

        updatedGroupIds.add(group.id!!)
        updatedGroupIds.addAll(getAllGroups(group.id!!)
            .map { it.groupId })

        identityRepository.updateIdentityModifiedAt(updatedGroupIds, Instant.now())
    }

    private fun toIdentityData(identityDataSpecification: Optional<ModifiableIdentityDataSpecification>):
        IdentityData? {
        return if (identityDataSpecification.isEmpty) {
            return null
        } else {
            val dataSpecification = identityDataSpecification.get()
            IdentityData(
                dataSpecification.slug?.orElse(null),
                dataSpecification.name?.orElse(null),
                dataSpecification.lastname?.orElse(null),
                dataSpecification.phone?.orElse(null),
                dataSpecification.email?.orElse(null),
                dataSpecification.additionalData?.orElse(null),
            )
        }
    }

    private fun mergeIdentityData(old: IdentityData?, spec: Optional<ModifiableIdentityDataSpecification>):
        IdentityData? {
        return if (spec.isEmpty) {
            old
        } else {
            val dataSpec = spec.get()
            IdentityData(
                mergeIdentityData(old?.slug, dataSpec.slug),
                mergeIdentityData(old?.name, dataSpec.name),
                mergeIdentityData(old?.lastname, dataSpec.lastname),
                mergeIdentityData(old?.phone, dataSpec.phone),
                mergeIdentityData(old?.email, dataSpec.email),
                mergeIdentityData(old?.additionalData, dataSpec.additionalData),
            )
        }
    }

    private fun mergeIdentityData(old: String?, new: Optional<String>?): String? {
        return if (new == null) {
            old
        } else {
            new.orElse(null)
        }
    }

    private fun getIdentityByIdSpecification(spec: IdentityIdOneOf): Identity {
        return if (spec.hasId()) {
            identityRepository.getByIdForUpdate(
                spec.id.get()
            )
        } else {
            val externalId = spec.externalId.get()
            identityRepository.getByExternalIdAndTypeIdForUpdate(
                externalId.externalId,
                externalId.typeId
            )
        }
    }

    private fun validateCreateRequest(spec: CreateIdentitySpecification): Pair<Identity?, IdentityType> {
        val identityExternalId = spec.identityExternalId
        val oldIdentity = if (identityExternalId.externalId != null) {
            identityRepository.findByExternalIdAndTypeId(
                identityExternalId.externalId,
                identityExternalId.typeId
            )
        } else {
            null
        }

        if (oldIdentity != null) {
            throw IdentityAlreadyExistsException(oldIdentity.externalId, oldIdentity.type.id)
        }

        val parent = if (spec.parentId.isPresent) {
            val idOneOf = spec.parentId.get()
            if (idOneOf.hasId()) {
                identityRepository.getByIdForUpdate(idOneOf.id.get())
            } else {
                val externalId = idOneOf.externalId.get()
                identityRepository.getByExternalIdAndTypeIdForUpdate(externalId.externalId, externalId.typeId)
            }
        } else {
            null
        }

        if (parent != null && !parent.type.isGroup) {
            throw IllegalArgumentException(String.format("%s is not a group", parent.id.toString()))
        }

        return Pair(parent, identityTypeRepository.getById(identityExternalId.typeId))
    }

    private fun getAllGroups(groupId: UUID): List<IdentityRelation> {
        return getAllPages(
            InnerListIdentityGroupsSpecification(
                null,
                INNER_PAGING_SIZE,
                IdentityRelation.ConnectionType.UNDEFINED,
                false
            ),
            { identityRelationRepository.findAllGroupsByIdentityIdAndSpec(groupId, it) },
            { it.identityId },
            { specification, token -> InnerListIdentityGroupsSpecification(specification, token) }
        )
    }

    private fun addAllChildrenGroups(id: UUID, groupIds: MutableSet<UUID>) {
        val allChildren = getAllPages(
            IdentityChildListSpecification(
                null,
                INNER_PAGING_SIZE,
                id
            ),
            { identityRepository.findAllChildrenByIdentityId(it) },
            { it.id },
            { specification, token -> IdentityChildListSpecification(specification, token) }
        )

        allChildren.forEach {
            groupIds.addAll(getAllGroups(it.id!!).map { identityRelation ->
                identityRelation.groupId
            })
        }
    }
}
