package ru.yandex.intranet.imscore.core.services.identityGroup.impl


import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Transactional
import ru.yandex.intranet.imscore.core.domain.identity.Identity
import ru.yandex.intranet.imscore.core.domain.identity.IdentityIdOneOf
import ru.yandex.intranet.imscore.core.domain.identity.specification.GetIdentityByExternalIdSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.GetIdentityByIdSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentityListSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.IdentityRelation
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.AddToGroupSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.ExistsInGroupSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.InnerListIdentityGroupsSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.ListIdentityGroupsSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.RemoveFromGroupSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.ReplaceGroupSpecification
import ru.yandex.intranet.imscore.core.exceptions.CycleException
import ru.yandex.intranet.imscore.core.exceptions.ResourceNotFoundException
import ru.yandex.intranet.imscore.core.exceptions.identity.IdentityIsNotGroupException
import ru.yandex.intranet.imscore.core.exceptions.identity.IdentityNotFoundException
import ru.yandex.intranet.imscore.core.ports.identity.IdentityRepository
import ru.yandex.intranet.imscore.core.ports.identityRelation.IdentityRelationRepository
import ru.yandex.intranet.imscore.core.services.identityGroup.IdentityGroupService
import ru.yandex.intranet.imscore.core.util.common.CommonUtils
import java.time.Instant
import java.util.UUID

/**
 *  Identity group service implementation
 *  The service is a special case of the identity relation with membership connection type
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Service
open class IdentityGroupServiceImpl(
    private val identityRepository: IdentityRepository,
    private val identityRelationRepository: IdentityRelationRepository,
): IdentityGroupService {

    @Transactional(readOnly = true)
    override fun listIdentityGroups(
        spec: ListIdentityGroupsSpecification
    ): List<IdentityRelation> {
        val identity = getIdentityBySpec(spec.identity, false)
        val identityRelations = identityRelationRepository.findAllGroupsByIdentityIdAndSpec(identity.id!!, spec)
        if (identityRelations.isEmpty()) {
            return identityRelations
        }
        val identities = getIdentityMapByIds(identityRelations.stream().map { it.groupId }.toList())
        identityRelations.forEach {
            it.group = identities[it.groupId]
        }
        return identityRelations
    }

    @Transactional(readOnly = true)
    override fun existsInGroup(spec: ExistsInGroupSpecification): Boolean {
        val identity = getIdentityBySpec(spec.identity, false)
        val group = getGroupIdentityBySpec(spec.group, false)
        return identityRelationRepository.existsIdentityRelationToGroup(
            identity.id!!, group.id!!, spec.connectionType, spec.onlyDirectly)
    }

    @Transactional
    override fun addToGroup(spec: AddToGroupSpecification) {
        val group = getGroupIdentityBySpec(spec.group, true)
        val identities = (getIdentitiesByIdSpec(spec.identities)).toSet()

        if (identities.contains(group)) {
            throw CycleException(group.id)
        }

        val identityRelations = mutableListOf<IdentityRelation>()
        for (identity in identities) {
            identityRelations.add(
                IdentityRelation(
                    identity.id!!,
                    group.id!!,
                    IdentityRelation.ConnectionType.MEMBERSHIP
                )
            )
        }
        identityRelationRepository.saveAll(identityRelations)

        val allGroups = mutableListOf<UUID>()
        allGroups.add(group.id!!)
        allGroups.addAll(getAllGroups(group.id!!)
            .map { it.groupId })
        identityRepository.updateIdentityModifiedAt(allGroups, Instant.now())
    }

    @Transactional
    override fun removeFromGroup(spec: RemoveFromGroupSpecification) {
        val group = getGroupIdentityBySpec(spec.group, true)
        val identities = getIdentitiesByIdSpec(spec.identities).toSet()

        val identityIds: List<UUID> = identities.stream().map{ it.id!! }.toList()
        identityRelationRepository.deleteByIdentityIdInAndGroupId(
            identityIds,
            group.id!!,
        )

        val identityIdsToDelete: List<UUID> = identities.stream()
            .filter{ it.parentId == group.id}
            .map{ it.id!! }
            .toList()
        identityRepository.deleteByIdIn(identityIdsToDelete)

        val allGroups = mutableListOf<UUID>()
        allGroups.add(group.id!!)
        allGroups.addAll(getAllGroups(group.id!!)
            .map { it.groupId })
        identityRepository.updateIdentityModifiedAt(allGroups, Instant.now())
    }

    @Transactional
    override fun replaceGroup(spec: ReplaceGroupSpecification) {
        val group = getGroupIdentityBySpec(spec.group, true)
        val identities = getIdentitiesByIdSpec(spec.identities).toSet()
        val identitysById = identities.associateBy { it.id }

        val identityRelations = identityRelationRepository.findAllDirectIdentityRelationByGroupId(group.id!!).toSet()
        val identityRelationByIdentityId = identityRelations.associateBy { it.identityId }

        val identityToAdd = mutableListOf<IdentityRelation>()
        val relationToRemove = mutableListOf<UUID>()
        val identityToRemove = mutableListOf<UUID>()

        for (identityRelation in identityRelations) {
            val identityId = identityRelation.identityId
            if (!identitysById.containsKey(identityId)) {
                if (identityRelation.connectionType == IdentityRelation.ConnectionType.OWNERSHIP) {
                    identityToRemove.add(identityId)
                } else {
                    relationToRemove.add(identityId)
                }
            }
        }

        for (identity in identities) {
            if (!identityRelationByIdentityId.containsKey(identity.id)) {
                identityToAdd.add(IdentityRelation(
                    identity.id!!,
                    group.id!!,
                    IdentityRelation.ConnectionType.MEMBERSHIP
                ))
            }
        }

        if (identityToAdd.isNotEmpty() || relationToRemove.isNotEmpty() || identityToRemove.isNotEmpty()) {
            identityRelationRepository.saveAll(identityToAdd)
            identityRepository.deleteByIdIn(identityToRemove)
            identityRelationRepository.deleteByIdentityIdInAndGroupIdAndConnectionType(
                relationToRemove, group.id!!,
                IdentityRelation.ConnectionType.MEMBERSHIP
            )

            val allGroups = mutableListOf<UUID>()
            allGroups.add(group.id!!)
            allGroups.addAll(getAllGroups(group.id!!)
                .map { it.groupId })
            identityRepository.updateIdentityModifiedAt(allGroups, Instant.now())
        }
    }

    private fun getGroupIdentityBySpec(idSpec: IdentityIdOneOf, forUpdate: Boolean): Identity {
        val group = getIdentityBySpec(idSpec, forUpdate)
        if (!group.type.isGroup) {
            throw IdentityIsNotGroupException(group.id!!)
        }
        return group
    }

    private fun getIdentityBySpec(idSpec: IdentityIdOneOf, forUpdate: Boolean): Identity {
        return if (!idSpec.hasId()) {
            val externalId = idSpec.externalId.get()
            if (forUpdate) {
                identityRepository.getByExternalIdAndTypeIdForUpdate(externalId.externalId, externalId.typeId)
            } else {
                identityRepository.getByExternalIdAndTypeId(
                    GetIdentityByExternalIdSpecification(
                        externalId.externalId,
                        externalId.typeId,
                        false
                    )
                )
            }
        } else {
            val id = idSpec.id.get()
            if (forUpdate) {
                identityRepository.getByIdForUpdate(id)
            } else {
                identityRepository.getById(
                    GetIdentityByIdSpecification(
                        id,
                        false
                    )
                )
            }
        }
    }

    private fun getIdentitiesByIdSpec(spec: List<IdentityIdOneOf>): List<Identity> {
        val identityIds = mutableListOf<UUID>()
        val externalIdMapByType = mutableMapOf<String, MutableList<String>>()
        spec.forEach {
            if (it.hasId()) {
                identityIds.add(it.id.get())
            } else {
                val externalId = it.externalId.get()
                externalIdMapByType.putIfAbsent(externalId.typeId, mutableListOf())
                externalIdMapByType[externalId.typeId]!!.add(externalId.externalId)
            }
        }

        var identities: MutableList<Identity> = identityRepository.findByIdInForUpdate(identityIds).toMutableList()
        for ((typeId, externalIds) in externalIdMapByType) {
            identities.addAll(
                identityRepository.findByExternalIdInAndTypeIdForUpdate(externalIds, typeId)
            )
        }
        identities = identities.distinct().toMutableList()

        if (identities.size != spec.size) {
            val identitiesMapById = identities.associateBy({it.id}, {it})
            val identitiesMapByExternalId = identities.filter{it.externalId != null}.associateBy({it.externalId}, {it})

            val errors = mutableListOf<IdentityNotFoundException>()
            spec.forEach {
                if (it.hasId()) {
                    val id = it.id.get()
                    if (!identitiesMapById.containsKey(id)) {
                        errors.add(IdentityNotFoundException(id))
                    }
                } else {
                    val externalId = it.externalId.get()
                    if (!identitiesMapByExternalId.containsKey(externalId.externalId)) {
                        errors.add(IdentityNotFoundException(externalId.externalId, externalId.typeId!!))
                    }
                }
            }
            throw ResourceNotFoundException("Resources not found", errors)
        }
        return identities
    }

    private fun getIdentityMapByIds(ids: List<UUID>): Map<UUID, Identity> {
        return identityRepository.getListBySpec(
            IdentityListSpecification(
                ids = ids,
                size = ids.size
            )
        ).associateBy { it.id!! }
    }

    private fun getAllGroups(groupId: UUID): List<IdentityRelation> {
        return CommonUtils.getAllPages(
            InnerListIdentityGroupsSpecification(
                null,
                CommonUtils.INNER_PAGING_SIZE,
                IdentityRelation.ConnectionType.UNDEFINED,
                false
            ),
            { identityRelationRepository.findAllGroupsByIdentityIdAndSpec(groupId, it) },
            { it.identityId },
            { specification, token -> InnerListIdentityGroupsSpecification(specification, token) }
        )
    }

}
