package ru.yandex.intranet.imscore.core.util.common;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Common utils
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public final class CommonUtils {
    private static final Logger LOG = LoggerFactory.getLogger(CommonUtils.class);
    private static final int INFINITY_LOOP_THRESHOLD = 10000;
    public static final int INNER_PAGING_SIZE = 500;

    private CommonUtils() {
    }

    /**
     * Get all pages as one List of T.
     *
     * Spec size expected to be {@link CommonUtils#INNER_PAGING_SIZE}, otherwise use
     * {@link #getAllPages(Object, Function, Function, BiFunction, Integer)} function.
     *
     * @param spec for getPageFunc
     * @param getPageFunc function tha should return List of T from given spec
     * @param getNextPageToken function that should return next page token from given T, or null
     * @param setNextPageTokenToSpec function that should set next page token to spec
     * @param <T> expected List elements
     * @param <U> spec
     * @param <R> next page token
     * @return List of T
     */
    public static <T, U, R> List<T> getAllPages(U spec,
                                                Function<U, List<T>> getPageFunc,
                                                Function<T, R> getNextPageToken,
                                                BiFunction<U, R, U> setNextPageTokenToSpec) {
        return getAllPages(spec, getPageFunc, getNextPageToken, setNextPageTokenToSpec, null);
    }

    /**
     * Get all pages as one List of T.
     *
     * @param spec for getPageFunc
     * @param getPageFunc function tha should return List of T from given spec
     * @param getNextPageToken function that should return next page token from given T, or null
     * @param setNextPageTokenToSpec function that should set next page token to spec
     * @param size expected size of page, must be greater than 1 or null,
     *             if null {@link #INNER_PAGING_SIZE} will be used otherwise
     * @param <T> expected List elements
     * @param <U> spec
     * @param <R> next page token
     * @return List of T
     */
    public static <T, U, R> List<T> getAllPages(U spec,
                                                Function<U, List<T>> getPageFunc,
                                                Function<T, R> getNextPageToken,
                                                BiFunction<U, R, U> setNextPageTokenToSpec,
                                                Integer size) {
        if (size != null && size < 2) {
            throw new IllegalArgumentException("size must be greater than 1!");
        }

        List<T> page = getPageFunc.apply(spec);
        List<T> result = new ArrayList<>(page);
        R nextPageToken;
        int expectedPageSize = size == null ? INNER_PAGING_SIZE : size;
        int i = 0;
        while (page.size() >= expectedPageSize
                && (nextPageToken = getNextPageToken.apply(page.get(page.size() - 1))) != null) {
            spec = setNextPageTokenToSpec.apply(spec, nextPageToken);
            page = getPageFunc.apply(spec);
            result.addAll(page);

            if (++i >= INFINITY_LOOP_THRESHOLD) {
                LOG.warn("Infinity loop detected in CommonUtils::getAllPages! Spec = [" + spec.toString() + "];");
                break;
            }
        }
        return result;
    }
}
