package ru.yandex.intranet.imscore.infrastructure.configurations;


import javax.sql.DataSource;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.transaction.TransactionManagerCustomizers;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.context.annotation.Primary;

import ru.yandex.intranet.imscore.infrastructure.configurations.datasources.ReadWriteRoutingDateSource;
import ru.yandex.intranet.imscore.infrastructure.configurations.datasources.RoutingDataSourceTransactionManager;


@Configuration
@EnableConfigurationProperties
public class PostgresRoutingConfiguration {
    public static final String WRITE_DATASOURCE_BEAN_NAME = "writeRoutingDataSource";

    public static final String READ_DATASOURCE_BEAN_NAME = "readRoutingDataSource";

    @Primary
    @Bean
    @ConfigurationProperties(prefix = "spring.datasource.routing.postgres.write")
    public HikariConfig writeDataSourceProperties() {
        return new HikariConfig();
    }

    @Bean
    @ConfigurationProperties(prefix = "spring.datasource.routing.postgres.read")
    public HikariConfig readDataSourceProperties() {
        return new HikariConfig();
    }

    @Bean(WRITE_DATASOURCE_BEAN_NAME)
    public DataSource writeDataSource(@Qualifier("writeDataSourceProperties") HikariConfig properties) {
        return new HikariDataSource(properties);
    }

    @Bean(READ_DATASOURCE_BEAN_NAME)
    public DataSource readDataSource(@Qualifier("readDataSourceProperties") HikariConfig properties) {
        return new HikariDataSource(properties);
    }

    @Primary
    @Bean
    @DependsOn({WRITE_DATASOURCE_BEAN_NAME, READ_DATASOURCE_BEAN_NAME})
    public DataSource routingDataSource(@Qualifier(WRITE_DATASOURCE_BEAN_NAME) DataSource writeDataSource,
                                        @Qualifier(READ_DATASOURCE_BEAN_NAME) DataSource readDataSource) {
        return new ReadWriteRoutingDateSource(writeDataSource, readDataSource);
    }

    @Bean(name = "transactionManager")
    public RoutingDataSourceTransactionManager routingDataSourceTransactionManager(
        DataSource dataSource,
        ObjectProvider<TransactionManagerCustomizers> transactionManagerCustomizersProvider
    ) {
        RoutingDataSourceTransactionManager transactionManager = new RoutingDataSourceTransactionManager();
        transactionManager.setDataSource(dataSource);
        TransactionManagerCustomizers transactionManagerCustomizers =
                transactionManagerCustomizersProvider.getIfAvailable();
        if (null != transactionManagerCustomizers) {
            transactionManagerCustomizers.customize(transactionManager);
        }
        return transactionManager;
    }

}
