package ru.yandex.intranet.imscore.infrastructure.data.converters.identity

import org.springframework.data.domain.Page
import ru.yandex.intranet.imscore.core.domain.identity.Identity
import ru.yandex.intranet.imscore.core.domain.identity.IdentityData
import ru.yandex.intranet.imscore.infrastructure.data.converters.identityData.IdentityDataConverters
import ru.yandex.intranet.imscore.infrastructure.data.converters.identityType.IdentityTypeConverters
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityBaseEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityWithDataEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.projection.IdentityWithDataFlatProjection
import ru.yandex.intranet.imscore.infrastructure.data.entities.identityType.IdentityTypeEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identityTypeSource.IdentityTypeSourceEntity
import java.util.UUID

/**
 * Identity entity converters
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
class IdentityConverters {
    companion object {
        fun toIdentity (from: IdentityBaseEntity): Identity {
            val data: IdentityData? = if (from is IdentityWithDataEntity && from.data != null) {
                IdentityDataConverters.toIdentityData(from.data!!)
            } else {
                null
            }

            return  Identity(
                id = from.id,
                parentId = from.parentId,
                externalId = from.externalId,
                type = IdentityTypeConverters.toIdentityType(from.type),
                data = data,
                createdAt = from.createdAt,
                modifiedAt = from.modifiedAt
            )
        }

        fun toIdentityEntity(from: Identity): IdentityEntity {
            val identityEntity = IdentityEntity()
            identityEntity.parentId = from.parentId
            identityEntity.externalId = from.externalId
            identityEntity.type = IdentityTypeEntity(
                from.type.id,
                isGroup = from.type.isGroup
            )

            identityEntity.id = from.id
            identityEntity.createdAt = from.createdAt
            return identityEntity
        }

        fun toIdentityWithDataEntity(from: Identity): IdentityWithDataEntity {
            val identityEntity = IdentityWithDataEntity()
            identityEntity.parentId = from.parentId
            identityEntity.externalId = from.externalId
            val source = IdentityTypeSourceEntity()
            source.id = from.type.sourceId
            identityEntity.type = IdentityTypeEntity(
                from.type.id,
                isGroup = from.type.isGroup,
                source = source,
            )

            identityEntity.id = from.id
            identityEntity.createdAt = from.createdAt

            if (from.data != null) {
                identityEntity.data = IdentityDataConverters.toIdentityDataEntity(
                    identityEntity,
                    from.data
                )
            }

            return identityEntity
        }

        private fun toIdentity(from: IdentityWithDataFlatProjection, withData: Boolean): Identity {
            val data = if (withData) {
                if (from.identityId != null) {
                    IdentityData(
                        from.slug,
                        from.name,
                        from.lastname,
                        from.phone,
                        from.email,
                        from.additionalData,
                    )
                } else {
                    null
                }
            } else {
                null
            }

            return Identity(
                id = UUID.fromString(from.id),
                parentId = if (from.parentId != null) {
                    UUID.fromString(from.parentId)
                } else {
                    null
                },
                externalId = from.externalId,
                type = IdentityTypeConverters.toIdentityType(from.type),
                data = data,
                createdAt = from.createdAt,
                modifiedAt = from.modifiedAt
            )
        }

        fun toIdentityList(from: Page<out IdentityBaseEntity>): List<Identity> {
            return from.map { toIdentity(it) }.toList()
        }

        @Suppress("REDUNDANT_PROJECTION")
        fun toIdentityList(from: List<out IdentityBaseEntity>): List<Identity> {
            return from.map { toIdentity(it) }.toList()
        }

        fun toIdentityListFromProjection(from: List<IdentityWithDataFlatProjection>, loadData: Boolean): List<Identity> {
            return from.map { toIdentity(it, loadData) }.toList()
        }
    }
}
