package ru.yandex.intranet.imscore.infrastructure.data.repositories.baseRepository;

import java.io.Serializable;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;

import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.support.JpaEntityInformation;
import org.springframework.data.jpa.repository.support.SimpleJpaRepository;
import org.springframework.data.support.PageableExecutionUtils;

import ru.yandex.intranet.imscore.infrastructure.data.repositories.baseRepository.jpa.BaseRepository;

/**
 * Custom implementation of the {@link org.springframework.data.repository.CrudRepository} interface with
 * {@link #findAll(Specification, long, int)} and {@link #findAll(Specification, long, int, Sort)}
 * and {@link #findAll(Specification, Pageable)}
 * without count request.
 *
 * @param <T> the type of the entity to handle
 * @param <ID> the type of the entity's identifier
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public class BaseRepositoryImpl<T, ID extends Serializable> extends SimpleJpaRepository<T, ID>
        implements BaseRepository<T, ID> {

    @SuppressWarnings("unused")
    private final EntityManager entityManager;

    public BaseRepositoryImpl(Class<T> domainClass, EntityManager entityManager) {
        super(domainClass, entityManager);
        this.entityManager = entityManager;
    }

    public BaseRepositoryImpl(JpaEntityInformation<T, ?> entityInformation, EntityManager entityManager) {
        super(entityInformation, entityManager);
        this.entityManager = entityManager;
    }

    @Override
    public List<T> findAll(Specification<T> spec, long offset, int maxResults) {
        return findAll(spec, offset, maxResults, Sort.unsorted());
    }

    @Override
    public List<T> findAll(Specification<T> spec, long offset, int maxResults, Sort sort) {
        TypedQuery<T> query = getQuery(spec, sort);

        if (offset < 0) {
            throw new IllegalArgumentException("Offset must not be less than zero!");
        }
        if (maxResults < 1) {
            throw new IllegalArgumentException("Max results must not be less than one!");
        }

        query.setFirstResult((int) offset);
        query.setMaxResults(maxResults);
        return query.getResultList();
    }

    @NotNull
    @Override
    public Page<T> findAll(Specification<T> spec, @NotNull Pageable pageable) {
        TypedQuery<T> query = getQuery(spec, pageable);
        return isUnpaged(pageable) ? new PageImpl<>(query.getResultList())
                : readPageWithoutCount(query, pageable);
    }

    private  <S extends T> Page<S> readPageWithoutCount(@NotNull TypedQuery<S> query,
                                                        Pageable pageable) {
        if (pageable.isPaged()) {
            query.setFirstResult((int) pageable.getOffset());
            query.setMaxResults(pageable.getPageSize());
        }

        return PageableExecutionUtils.getPage(query.getResultList(), pageable,
                () -> Long.MAX_VALUE);
    }

    private static boolean isUnpaged(Pageable pageable) {
        return pageable.isUnpaged();
    }

}
