package ru.yandex.intranet.imscore.infrastructure.data.repositories.identity

import org.springframework.data.domain.PageRequest
import org.springframework.data.domain.Sort
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Propagation
import org.springframework.transaction.annotation.Transactional
import ru.yandex.intranet.imscore.core.domain.identity.Identity
import ru.yandex.intranet.imscore.core.domain.identity.specification.GetIdentityByExternalIdSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.GetIdentityByIdSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentityChildListSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentityListSpecification
import ru.yandex.intranet.imscore.core.exceptions.identity.IdentityNotFoundException
import ru.yandex.intranet.imscore.core.ports.identity.IdentityRepository
import ru.yandex.intranet.imscore.infrastructure.data.converters.identity.IdentityConverters
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityBaseEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityWithDataEntity
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identity.jpa.JpaIdentityRepository
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identity.jpa.JpaIdentityWithDataFlatProjectionRepository
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identity.jpa.JpaIdentityWithDataRepository
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identity.specifications.IdentitySpecifications
import java.time.Instant
import java.util.Optional
import java.util.UUID

/**
 * Identity repository implementation
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Repository
open class IdentityRepositoryImpl(
    private val jpaIdentityRepository: JpaIdentityRepository,
    private val jpaIdentityWithDataRepository: JpaIdentityWithDataRepository,
    private val jpaIdentityWithDataFlatProjectionRepository: JpaIdentityWithDataFlatProjectionRepository,
): IdentityRepository {

    override fun findByIdInForUpdate(ids: List<UUID>): List<Identity> {
        return IdentityConverters.toIdentityList(
            jpaIdentityRepository.findByIdInForUpdate(ids)
        )
    }

    override fun getListBySpec(spec: IdentityListSpecification): List<Identity> {
        val pageable = PageRequest.of(0, spec.size, Sort.Direction.ASC, "id")
        val page = if (spec.loadData) {
            val jpaSpec = IdentitySpecifications.fetchData<IdentityWithDataEntity>()
                .and(IdentitySpecifications.greaterId<IdentityWithDataEntity>(spec.cursor)
                    .and(IdentitySpecifications.byIds(spec.ids)))
            jpaIdentityWithDataRepository.findAll(jpaSpec, pageable)
        } else {
            val jpaSpec = IdentitySpecifications.greaterId<IdentityEntity>(spec.cursor)
                .and(IdentitySpecifications.byIds(spec.ids))
            jpaIdentityRepository.findAll(jpaSpec, pageable)
        }

        return IdentityConverters.toIdentityList(page)
    }

    override fun getById(spec: GetIdentityByIdSpecification): Identity {
        val entity: Optional<out IdentityBaseEntity> = if (spec.withData) {
            jpaIdentityWithDataRepository.findByIdWithData(spec.id)
        } else {
            jpaIdentityRepository.findById(spec.id)
        }
        val identity = entity.orElseThrow {
            IdentityNotFoundException(spec.id)
        }

        return IdentityConverters.toIdentity(identity)
    }

    @Transactional(propagation = Propagation.REQUIRED)
    override fun getByIdForUpdate(id: UUID): Identity {
        return jpaIdentityRepository
            .findByIdForUpdate(id)
            .map {
                IdentityConverters.toIdentity(it)
            }.orElseThrow { IdentityNotFoundException(id) }
    }

    override fun findByExternalIdInAndTypeId(externalIds: List<String>, typeId: String): List<Identity> {
        val entities = jpaIdentityRepository.findByExternalIdInAndTypeId(externalIds, typeId)
        return IdentityConverters.toIdentityList(entities)
    }

    override fun findByExternalIdInAndTypeIdForUpdate(externalIds: List<String>, typeId: String): List<Identity> {
        val entities = jpaIdentityRepository.findByExternalIdInAndTypeIdForUpdate(externalIds, typeId)
        return IdentityConverters.toIdentityList(entities)
    }

    override fun findByExternalIdAndTypeId(externalId: String, typeId: String): Identity? {
        return jpaIdentityWithDataRepository.findByExternalIdAndTypeIdWithData(externalId, typeId)
            .map {
                IdentityConverters.toIdentity(it)
            }.orElse(null)
    }

    override fun getByExternalIdAndTypeId(spec: GetIdentityByExternalIdSpecification): Identity {
        val entity: Optional<out IdentityBaseEntity> = if (spec.withData) {
            jpaIdentityWithDataRepository.findByExternalIdAndTypeIdWithData(spec.externalId, spec.typeId)
        } else {
            jpaIdentityRepository.findByExternalIdAndTypeId(spec.externalId, spec.typeId)
        }

        val identity = entity.orElseThrow {
            IdentityNotFoundException(spec.externalId, spec.typeId)
        }

        return IdentityConverters.toIdentity(identity)
    }

    @Transactional(propagation = Propagation.REQUIRED)
    override fun findByExternalIdAndTypeIdForUpdate(externalId: String, typeId: String): Identity? {
        return jpaIdentityRepository
            .findByExternalIdAndTypeIdForUpdate(externalId, typeId)
            .map {
                IdentityConverters.toIdentity(it)
            }.orElse(null)
    }

    @Transactional(propagation = Propagation.REQUIRED)
    override fun getByExternalIdAndTypeIdForUpdate(externalId: String, typeId: String): Identity {
        return jpaIdentityRepository
            .findByExternalIdAndTypeIdForUpdate(externalId, typeId)
            .map {
                IdentityConverters.toIdentity(it)
            }.orElseThrow {
                IdentityNotFoundException(externalId, typeId)
            }
    }

    override fun save(model: Identity): Identity {
        val identityEntity = IdentityConverters.toIdentityWithDataEntity(model)
        return IdentityConverters.toIdentity(jpaIdentityWithDataRepository.saveAndFlush(identityEntity))
    }

    override fun findAllIdentitiesByGroupId(spec: IdentityListSpecification): List<Identity> {
        val page = jpaIdentityWithDataFlatProjectionRepository.findAllIdentitiesByGroupId(spec.groupId,
            spec.onlyDirectly,
            spec.loadData,
            spec.cursor,
            spec.size,
        )

        return IdentityConverters.toIdentityListFromProjection(page, spec.loadData)
    }

    override fun findAllChildrenByIdentityId(spec: IdentityChildListSpecification): List<Identity> {
        val page = jpaIdentityRepository.findAllChildrenByIdentityId(
            spec.groupId,
            spec.cursor,
            spec.size,
        )

        return IdentityConverters.toIdentityList(page)
    }

    override fun deleteById(id: UUID): Int {
        val count = jpaIdentityRepository.deleteByIdAndReturnCount(id)
        if (count <= 0) {
            throw IdentityNotFoundException(id)
        }
        return count
    }

    override fun deleteByExternalId(externalId: String, typeId: String): Int {
        val count = jpaIdentityRepository.deleteByExternalIdAndReturnCount(externalId, typeId)
        if (count <= 0) {
            throw IdentityNotFoundException(externalId, typeId)
        }
        return count
    }

    override fun deleteByIdIn(ids: List<UUID>) {
        jpaIdentityRepository.deleteByIdIn(ids)
    }

    @Transactional(propagation = Propagation.REQUIRED)
    override fun updateIdentityModifiedAt(ids: List<UUID>, modifiedAt: Instant) {
        jpaIdentityRepository.updateIdentityModifiedAt(ids, modifiedAt)
    }
}
