package ru.yandex.intranet.imscore.infrastructure.data.repositories.identity.jpa

import org.springframework.data.domain.Page
import org.springframework.data.domain.Pageable
import org.springframework.data.jpa.domain.Specification
import org.springframework.data.jpa.repository.JpaSpecificationExecutor
import org.springframework.data.jpa.repository.Lock
import org.springframework.data.jpa.repository.Modifying
import org.springframework.data.jpa.repository.Query
import org.springframework.stereotype.Repository
import ru.yandex.intranet.imscore.infrastructure.data.entities.identity.IdentityEntity
import ru.yandex.intranet.imscore.infrastructure.data.repositories.baseRepository.jpa.BaseRepository
import java.time.Instant
import java.util.Optional
import java.util.UUID
import javax.persistence.LockModeType

/**
 * Identity jpa repository interface
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Repository
interface JpaIdentityRepository:
    BaseRepository<IdentityEntity, UUID>,
    JpaSpecificationExecutor<IdentityEntity> {

    override fun findAll(spec: Specification<IdentityEntity>?, pageable: Pageable): Page<IdentityEntity>

    @Lock(LockModeType.PESSIMISTIC_WRITE)
    @Query("""
        SELECT i
        FROM IdentityEntity i
        WHERE i.id = :id
        ORDER BY i.id""")
    fun findByIdForUpdate(id: UUID): Optional<IdentityEntity>

    @Lock(LockModeType.PESSIMISTIC_WRITE)
    @Query("""
        SELECT i
        FROM IdentityEntity i
        WHERE i.id in (:ids)
        ORDER BY i.id""")
    fun findByIdInForUpdate(ids: List<UUID>): List<IdentityEntity>

    @Query("""
        WITH ids as (
            SELECT id
            FROM identity
            WHERE
                identity.external_id = :externalId
                AND identity.type_id = :typeId
        )
        SELECT *
        FROM identity
        WHERE identity.id = (SELECT id FROM ids)
        ORDER BY identity.id
        FOR UPDATE""", nativeQuery = true)
    fun findByExternalIdAndTypeIdForUpdate(externalId: String, typeId: String): Optional<IdentityEntity>

    @Query("""
        WITH ids as (
            SELECT id
            FROM identity
            WHERE
                identity.external_id IN (:externalIds)
                AND identity.type_id = :typeId
        )
        SELECT *
        FROM identity
        WHERE identity.id IN (SELECT id FROM ids)
        ORDER BY identity.id
        FOR UPDATE""", nativeQuery = true)
    fun findByExternalIdInAndTypeIdForUpdate(externalIds: List<String>, typeId: String): List<IdentityEntity>

    @Query("SELECT i FROM IdentityEntity i WHERE i.externalId = :externalId AND i.type.id = :typeId")
    fun findByExternalIdAndTypeId(externalId: String, typeId: String): Optional<IdentityEntity>

    @Query("select i from IdentityEntity i where i.externalId IN (:externalIds) and i.type.id=:typeId")
    fun findByExternalIdInAndTypeId(externalIds: List<String>, typeId: String): List<IdentityEntity>

    @Query(value = """WITH deleted AS (
        DELETE FROM identity i WHERE i.id = :id RETURNING *)
        SELECT count(*) FROM deleted""", nativeQuery = true)
    fun deleteByIdAndReturnCount(id: UUID): Int

    @Query(value = """WITH deleted AS (
        DELETE FROM identity i WHERE i.external_id = :externalId AND i.type_id = :typeId RETURNING *)
        SELECT count(*) FROM deleted""", nativeQuery = true)
    fun deleteByExternalIdAndReturnCount(externalId: String, typeId: String): Int

    @Suppress("SpringDataRepositoryMethodParametersInspection")
    fun deleteByIdIn(ids: List<UUID>)

    @Modifying(clearAutomatically = true)
    @Query(value = """
        UPDATE identity
        SET modified_at =  :modifiedAt
        WHERE identity.id IN (:ids)
        AND identity.modified_at < :modifiedAt
    """, nativeQuery = true)
    fun updateIdentityModifiedAt(ids: List<UUID>, modifiedAt: Instant)

    @Query(value = """
        WITH RECURSIVE
            r AS (SELECT i.id
                  FROM identity i
                  WHERE i.parent_id = :groupId
                  UNION
                  SELECT ir.id
                  FROM r
                  LEFT JOIN identity ir ON ir.parent_id = r.id),
            identitys AS (SELECT DISTINCT r.id
                          FROM r
                          WHERE (:cursor IS NULL
                              OR
                                 (r.id > cast(cast(:cursor AS text) AS uuid))
                                    )
                          ORDER BY r.id
                          LIMIT :limit)
        SELECT *
        FROM identity
        WHERE identity.id in (SELECT identitys.id FROM identitys)
        ORDER BY identity.id
    """, nativeQuery = true)
    fun findAllChildrenByIdentityId(groupId: UUID, cursor: UUID?, limit: Int): List<IdentityEntity>

}
