package ru.yandex.intranet.imscore.infrastructure.data.repositories.identityRelation

import org.springframework.stereotype.Repository
import ru.yandex.intranet.imscore.core.domain.identityRelation.IdentityRelation
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.IdentityRelationListSpecification
import ru.yandex.intranet.imscore.core.ports.identityRelation.IdentityRelationRepository
import ru.yandex.intranet.imscore.infrastructure.data.converters.identityRelation.IdentityRelationConverters
import ru.yandex.intranet.imscore.infrastructure.data.entities.identityRelation.IdentityRelationEntity
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identityRelation.jpa.JpaIdentityRelationRepository
import java.util.UUID

/**
 * IdentityRelation repository implementation
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Repository
open class IdentityRelationRepositoryImpl(
    private val jpaIdentityRelationRepository: JpaIdentityRelationRepository,
): IdentityRelationRepository {

    override fun findAllGroupsByIdentityIdAndSpec(
        identityId: UUID, spec: IdentityRelationListSpecification
    ): List<IdentityRelation> {
        val entities = jpaIdentityRelationRepository.findAllGroupsByIdentityId(
            identityId,
            IdentityRelationEntity.ConnectionType.fromInt(spec.connectionType.value),
            spec.onlyDirectly,
            spec.cursor, spec.size
        )
        return IdentityRelationConverters.toIdentityRelationList(entities)
    }

    override fun existsIdentityRelationToGroup(
        identityId: UUID, groupId: UUID, connectionType: IdentityRelation.ConnectionType, onlyDirectly: Boolean
    ): Boolean {
        return jpaIdentityRelationRepository.existsIdentityRelationToGroup(
            identityId, groupId, IdentityRelationEntity.ConnectionType.fromInt(connectionType.value), onlyDirectly)
    }

    override fun saveAll(models: List<IdentityRelation>) {
        val entities = models.stream().map { IdentityRelationConverters.toIdentityRelationEntity(it) }.toList()
        entities.forEach{
            jpaIdentityRelationRepository.upsert(it.identityId, it.groupId, it.connectionType)
        }
    }

    override fun deleteByIdentityIdInAndGroupIdAndConnectionType(
        identityIds: List<UUID>, groupId: UUID, connectionType: IdentityRelation.ConnectionType
    ) {
        jpaIdentityRelationRepository.deleteByIdentityIdInAndGroupIdAndConnectionType(
            identityIds, groupId, IdentityRelationEntity.ConnectionType.fromInt(connectionType.value))
    }

    override fun deleteByIdentityIdInAndGroupId(identityIds: List<UUID>, groupId: UUID) {
        jpaIdentityRelationRepository.deleteByIdentityIdInAndGroupId(identityIds, groupId)
    }

    override fun save(identityRelation: IdentityRelation): IdentityRelation {
        val identityRelationEntity = jpaIdentityRelationRepository.save(
            IdentityRelationConverters.toIdentityRelationEntity(identityRelation))
        return IdentityRelationConverters.toIdentityRelation(identityRelationEntity)
    }

    override fun upsert(identityRelation: IdentityRelation) {
        jpaIdentityRelationRepository.upsert(
            identityRelation.identityId,
            identityRelation.groupId,
            IdentityRelationEntity.ConnectionType.fromInt(identityRelation.connectionType.value)
        )
    }

    override fun findAllDirectIdentityRelationByGroupId(groupId: UUID): List<IdentityRelation> {
        val entities = jpaIdentityRelationRepository.findAllDirectIdentityRelationByGroupId(groupId)
        return IdentityRelationConverters.toIdentityRelationList(entities)
    }
}
